<?php

use bff\db\Dynprops;
use bff\db\Publicator;
use bff\utils\Files;

# таблицы
define('TABLE_AFISHA_TYPES',           DB_PREFIX.'afisha_types');
define('TABLE_AFISHA_ITEMS_',          DB_PREFIX.'afisha_items_');
define('TABLE_AFISHA_ITEMS_LANG',      DB_PREFIX.'afisha_items_lang');
define('TABLE_AFISHA_COMMENTS',        DB_PREFIX.'afisha_comments');
define('TABLE_AFISHA_PLACES',          DB_PREFIX.'afisha_places');
define('TABLE_AFISHA_TAGS',            DB_PREFIX.'afisha_tags');
define('TABLE_AFISHA_TAGS_IN',         DB_PREFIX.'afisha_tags_in');
define('TABLE_AFISHA_CATEGORIES',      DB_PREFIX.'afisha_categories');
define('TABLE_AFISHA_CATEGORIES_LANG', DB_PREFIX.'afisha_categories_lang');
define('TABLE_AFISHA_CATEGORIES_IN',   DB_PREFIX.'afisha_categories_in');
define('TABLE_AFISHA_CALENDAR',        DB_PREFIX.'afisha_calendar');
define('TABLE_AFISHA_DYNPROPS',        DB_PREFIX.'afisha_dynprops');
define('TABLE_AFISHA_DYNPROPS_MULTI',  DB_PREFIX.'afisha_dynprops_multi');
define('TABLE_AFISHA_DYNPROPS_IN',     DB_PREFIX.'afisha_dynprops_in');

/**
 * Модуль "Afisha"
 * настройки, сохраняемые в config:
 *   1) afisha_items_mod_{key} - счетчик непромодерированных событий указанного типа (добавляемые пользователями)
 *   2) afisha_comments_mod_all - общий счетчик непромодерированных комментариев (всех типов, с модерацией комментариев)
 */

abstract class AfishaBase extends Module
{
    /** @var AfishaModel */
    public $model = null;
    protected $securityKey = 'c28070fa2fe5d6c1734f77b48c757d27';
    
    /** @var bool компактное меню в админ панели */
    var $menuMini = true;

    # типы контента
    const contentTypePhoto = 1;
    const contentTypeVideo = 2;
    
    # тип пользователя (при добавлении события)
    const USERTYPE_UNKNOWN   = 0; // при добавлении из админ-панели
    const USERTYPE_MEMBER    = 1; // Пользователь
    const USERTYPE_ORGANIZER = 2; // Организатор

    # типы событий
    const TYPE_MOVIE    = 1; # Кино
    const TYPE_THEATER  = 2; # Театр
    const TYPE_EXHIBITION = 3; # Выставки
    const TYPE_CONCERT  = 4; # Концерты
    const TYPE_ACTIVITY = 5; # Мероприятия
    const TYPE_SPORT    = 6; # Спорт
    const TYPE_PARTY    = 7; # Вечеринки
    const TYPE_CIRCUS   = 8; # Цирк

    /** ID дин. свойства "жанр (фильтр)" */
    const MOVIE_GENRE_DP_ID = 17;

    /** @var AfishaSettings объект настроек текущего типа */
    var $type = null;
    
    # ---------------------------------------------------------------------------------------------
    # Записи

    public function init()
    {
        parent::init();

        $this->module_title = 'Афиша';

        bff::autoloadEx(array(
            'AfishaSettings' => array('app', 'modules/afisha/afisha.settings.php'),
            'AfishaComments' => array('app', 'modules/afisha/afisha.comments.php'),
            'AfishaImages'   => array('app', 'modules/afisha/afisha.images.php'),
            'AfishaVideo'    => array('app', 'modules/afisha/afisha.video.php'),
            'AfishaTags'     => array('app', 'modules/afisha/afisha.tags.php'),
        ));

        if (bff::adminPanel()) {
            if (strpos(bff::$event, 'dynprops') !== false) {
                $this->dp();
            }
        }
    } 

    /**
     * Shortcut
     * @return Afisha
     */
    public static function i()
    {
        return bff::module('afisha');
    }

    /**
     * Shortcut
     * @return AfishaModel
     */
    public static function model()
    {
        return bff::model('afisha');
    }

    /**
     * Формирование URL
     * @param string $key ключ
     * @param mixed $opts параметры
     * @param boolean $dynamic динамическая ссылка
     * @return string
     */
    public static function url($key = '', $opts = array(), $dynamic = false)
    {
        $base = static::urlBase(LNG, $dynamic);
        switch ($key)
        {
            # Главная
            case 'index':
                $base .= '/'.static::urlQuery($opts);
            break;
            # Просмотр события
            case 'view':
                if (is_string($opts)) {
                    return strtr($opts, array(
                        '{sitehost}' => SITEHOST . bff::locale()->getLanguageUrlPrefix(),
                    ));
                }
            break;
             # Список событий
            case 'list':
                if (!is_array($opts)) {
                    $type = intval($opts);
                    $opts = array();
                } else {
                    $type = 0;
                    if (!empty($opts['type'])) {
                        $type = intval($opts['type']);
                    }
                }
                return $base.'/'.static::i()->getTypeKeywordByID($type).'/' . static::urlQuery($opts, array('type'));
                break;
             # Список событий, подраздел
            case 'list.tab':
                return $base.'/'.
                    static::i()->getTypeKeywordByID($opts['type']).'/'.
                    ( ! empty($opts['tab']) ? $opts['tab'] : '').
                    static::urlQuery($opts, array('type', 'tab'));
                break;
             # Список мест
            case 'places':
                if (!is_array($opts)) {
                    $type = intval($opts);
                    $opts = array();
                } else {
                    $type = 0;
                    if (!empty($opts['type'])) {
                        $type = intval($opts['type']);
                    }
                }
                return $base.'/'.
                    static::i()->getTypeKeywordByID($type).'/places'.
                    static::urlQuery($opts, array('type'));
                break;
            # Страница события
            case 'place':
                if ( ! empty($opts['type'])) {
                    $base .= '/'.static::i()->getTypeKeywordByID($opts['type']);
                }
                $base .= '/places';
                if ( ! empty($opts['link'])) {
                    $base .= '/'.$opts['link'];
                }
                $base .= static::urlQuery($opts, array('type', 'link'));
            break;
            # Форма добавления события
            case 'add':
                $base .= '/add.html'.static::urlQuery($opts);
            break;
        }
        return $base;
    }

    /**
     * Корректируем динамическую ссылку в заголовке
     * @param string $title @ref
     * @return string
     */
    public static function urlInTitle(&$title)
    {
        return strtr($title, array(
            '{sitehost}' => SITEHOST . bff::locale()->getLanguageUrlPrefix(),
        ));
    }

    /**
     * Описание seo шаблонов страниц
     * @return array
     */
    public function seoTemplates()
    {
        $dateMacros = array(
            'date' => array('t' => 'Дата ('.tpl::date_format2(978300000, false, true, ' ', ', ', true).')'),
            'date2' => array('t' => 'Дата ('.tpl::date_format2(978300000, false, false).')'),
            'date3' => array('t' => 'Дата ('.tpl::dateFormat(978300000).')'),
        );
        $aTemplates = array(
            'groups' => array(
                'def' => array('t'=>'Общие'),
            ),
            'pages' => array(
                'index' => array( // index
                    't'      => 'Главная страница',
                    'group'  => 'def',
                    'macros' => array_merge(array(
                        'region' => array('t' => 'Регион'),
                    ), $dateMacros),
                    'fields' => array(
                        'titleh1' => array(
                            't'      => 'Заголовок H1',
                            'type'   => 'text',
                        ),
                    ),
                ),
                'add' => array( // add
                    't'      => 'Добавление события',
                    'group'  => 'def',
                    'macros' => array(
                        'region' => array('t' => 'Регион'),
                    ),
                    'fields' => array(
                        // ...
                    ),
                ),
                'view' => array( // view
                    't'      => 'Просмотр события',
                    'inherit'=> true,
                    'group'  => 'def',
                    'macros' => array(
                        'title'  => array('t' => 'Заголовок'),
                        'description'  => array('t' => 'Описание (до 150 символов)'),
                        'region' => array('t' => 'Регион'),
                    ),
                    'fields' => array(
                        'share_title'       => array(
                            't'    => 'Заголовок (поделиться в соц. сетях)',
                            'type' => 'text',
                        ),
                        'share_description' => array(
                            't'    => 'Описание (поделиться в соц. сетях)',
                            'type' => 'textarea',
                        ),
                        'share_sitename'    => array(
                            't'    => 'Название сайта (поделиться в соц. сетях)',
                            'type' => 'text',
                        ),
                    ),
                ),
            ),
        );

        foreach ($this->getTypes() as $type)
        {
            $typeID = $type->id;
            $aTemplates['groups'][$typeID] = array('t'=>$type->title);
            if ($typeID == self::TYPE_MOVIE) {
                $aTemplates['pages']['list-day-'.$typeID] = array( // listing_day
                    't'      => 'Список фильмов: за день',
                    'group'  => $typeID,
                    'macros' => array(
                        'region' => array('t' => 'Регион'),
                    ),
                    'fields' => array(
                        // ...
                    ),
                );
                $aTemplates['pages']['list-soon-'.$typeID] = array( // listing_soon
                    't'      => 'Список фильмов: скоро',
                    'group'  => $typeID,
                    'macros' => array(
                        'region' => array('t' => 'Регион'),
                    ),
                    'fields' => array(
                        // ...
                    ),
                );
                $aTemplates['pages']['list-archive-'.$typeID] = array( // listing_archive
                    't'      => 'Список фильмов: архив',
                    'list'   => true,
                    'group'  => $typeID,
                    'macros' => array(
                        'region' => array('t' => 'Регион'),
                    ),
                    'fields' => array(
                        // ...
                    ),
                );
            } else {
                $aTemplates['pages']['list-period-'.$typeID] = array( // listing_period
                    't'      => 'Список событий: период',
                    'group'  => $typeID,
                    'macros' => array(
                        'region' => array('t' => 'Регион'),
                        'period-from' => array('t' => 'Период: от'),
                        'period-to' => array('t' => 'Период: до'),
                    ),
                    'fields' => array(
                        // ...
                    ),
                );
            }
            $aTemplates['pages']['places-'.$typeID] = array( // places
                't'      => 'Список мест',
                'group'  => $typeID,
                'macros' => array(
                    'region' => array('t' => 'Регион'),
                ),
                'fields' => array(
                    // ...
                ),
            );
            $aTemplates['pages']['place-'.$typeID] = array( // places_view
                't'      => 'Просмотр места события',
                'group'  => $typeID,
                'macros' => array(
                    'title'  => array('t' => 'Заголовок'),
                    'description' => array('t' => 'Описание (до 150 символов)'),
                    'region' => array('t' => 'Регион'),
                ),
                'fields' => array(
                    // ...
                ),
            );
            if ($typeID == self::TYPE_MOVIE) {
                foreach (array('list-day-'.$typeID, 'place-'.$typeID) as $k) {
                    $aTemplates['pages'][$k]['macros'] = array_merge($aTemplates['pages'][$k]['macros'], $dateMacros);
                }
            }
        }

        return $aTemplates;
    }

    /**
     * Обработка данных события 
     * @param mixed $nItemID ID записи
     * @return array
     */
    protected function validateItemData($nItemID = 0)
    {   
        $aParams = array(
            'link'          => TYPE_STR,  # ссылка для заголовка
            'category_id'   => TYPE_UINT, # id категории
            'country_id'    => TYPE_UINT, # id страны
            'city_id'       => TYPE_UINT, # id города
            'user_id'       => TYPE_UINT, # id пользователя
            'img'           => TYPE_STR,  # имя файла изображения
            'img_crop'      => TYPE_STR,  # настройки crop'а изображения
            'content_type'  => TYPE_UINT, # ! тип контента
            'enabled'       => TYPE_BOOL, # публиковать
            'main'          => TYPE_BOOL, # на главной
            'extra'         => TYPE_ARRAY, # доп. параметры
            'd'             => TYPE_ARRAY, # дин. свойства
            'mtemplate'     => TYPE_BOOL,  # использовать общий шаблон SEO
        );
        if ($this->type->contentPublicator()) {
            $aParams['content'] = TYPE_ARRAY;
            unset($this->model->langItems['content']);
        }

        $aData = $this->input->postm($aParams);
        $this->input->postm_lang($this->model->langItems, $aData );

        $aParamsExtra = array(
            'source_link' => TYPE_ARRAY_STR,
            'xml_yandex'  => TYPE_BOOL,
        );
        switch ($this->type->id)
        {
            case 0: {
                /** 
                * add extra settings by typeID, 
                * @example: $aParamsExtra['test'] = TYPE_STR;
                */
            } break;
        }
        
        $this->input->clean_array($aData['extra'], $aParamsExtra);
        
        if (Request::isPOST())
        {
            if ($nItemID) {
                $this->prepareItemTitle($nItemID, $aData);
            }

            foreach ($aData['title'] as $lng => $t) {
                $aData['title_alt'][$lng] = HTML::escape(strip_tags(strtr($t, array('[link]'=>'','[/link]'=>''))));
            }

            if ($nItemID>0 && $this->type->tagsEnabled()) {
                $this->initTags($this->type->id)->tagsSave($nItemID);
            }
            
            foreach ($aData['extra'] as $k=>$v) {
                $aData['extra_'.$k] = $v;
            } unset($aData['extra']);

            if ($this->type->extra['source_link']) {
                $this->input->clean_array($aData['extra_source_link'], array(
                    'link'  => TYPE_STR,
                    'title' => TYPE_STR,
                ));
                $aData['extra_source_link'] = serialize($aData['extra_source_link']);
            } else {
                unset($aData['extra_source_link']);
            }
            
            if ( ! $this->type->extra['main']) {
                unset($aData['main']);
            }

            if ($this->type->contentPublicator()) {
                $aDataPublicator = $this->initPublicator()->dataPrepare($aData['content'], $nItemID);
                $aData['content'] = $aDataPublicator['content'];
                $aData['content_search'] = $aDataPublicator['content_search'];
                $aData['content_type'] = $this->updateContentType($aDataPublicator);
            }

            # на случай если настройки типа были изменены в процессе, 
            # оставляем неиспользуемые данные БЕЗ изменений
            $unsetData = array(
                //'regions' => array('country_id', 'city_id'),
                'content_short' => array('content_short'),
                'categories' => array('category_id'),
                'img' => array('img','img_text'),
                'author' => array('user_id','author'),
            );
            foreach ($unsetData as $k=>$v) {
                if ( ! $this->type->$k) {
                    if (is_string($v)) $v = array($v);
                    foreach ($v as $v2) {
                        unset($aData[$v2]);
                    }
                }
            }
        } else {
            if ( ! $nItemID) {
                foreach ($aData['extra'] as $k=>$v) {
                    $aData['extra_'.$k] = $v;
                } unset($aData['extra']);

                if ($this->type->extra['source_link']) {
                    $this->input->clean_array($aData['extra_source_link'], array(
                        'link' => TYPE_STR,
                        'title' => TYPE_STR,
                    ));
                }
                
                if ($this->type->authorUser() && !isset($aData['user_login'])) {
                    $aData['user_login'] = '';
                }

                $aData['mtemplate'] = 1;
            }
        }
        
        return $aData;
    }
    
    function saveItemPlace($nItemID, $nItemCityID, $bEdit)
    {
        if ( ! $nItemID) return false;
        
        $sqlPlacesTable = TABLE_AFISHA_PLACES;
        
        $sqlDelete = array();
        
        $aPlaces = array();
        $aPlacesID = array();
        $aInput = $this->input->post('place', TYPE_ARRAY_ARRAY);
        foreach ($aInput as $k=>$v)
        {
            $this->input->clean_array($v, array(
                'place_id'    => TYPE_UINT,
                'place_title' => TYPE_NOTAGS,
                'periods'     => TYPE_ARRAY,
            ));


            if ($this->type->placesMany() && !$v['place_id']) {
                // запрещаем не привязывать к объектам
                continue;
            }
            if ($v['place_id'] > 0) {
                if ( in_array($v['place_id'], $aPlacesID) ) {
                    continue; //место уже было привязано выше
                }
                $v['place_title'] = '';
            }
            
            if (empty($v['periods'])) continue;
            $placePeriods = array();
             
            foreach ($v['periods'] as $pk=>$pv)
            {
                $this->input->clean_array($pv, array(
                    'id'     => TYPE_UINT,
                    'period' => TYPE_STR,
                    'time'   => ( $this->type->placesTimeComaSeparated() ? TYPE_ARRAY_STR : TYPE_STR ),
                ));
                
                // дата или время указаны некорректно
                if (empty($pv['period'])) { continue; }
                // if ($this->type->placesTimeComaSeparated() && empty($pv['time'])) continue;
                
                # проверяем дату:
                $from = false; $to = false;
                if (strpos($pv['period'], ' - ') !== false) // период
                {
                    list($from, $to) = explode(' - ', $pv['period']);
                    $from = strtotime($from);
                    $to = strtotime($to);
                } else {
                    $from = strtotime($pv['period']);
                }
                if ($from === false || $from === -1) continue; // дата начала периода указана некорректно
                if ($to === false || $to === -1) { // дата завершения периода указана некорректно
                    $to = false;
                }
                
                if ($to!==false) { // несколько дней
                    if ($from > $to) continue; // период указан некорректно
                    foreach ($placePeriods as $sel) { // указанный день уже был задействован ранее
                        if ( ($from >= $sel['from'] && $from <= $sel['to']) ||
                            ($to >= $sel['from'] && $to <= $sel['to']) ) { 
                                continue 2;
                            }
                    }
                } else { // 1 день
                    foreach ($placePeriods as $sel) { // указанный день уже был задействован ранее
                        if ( $from >= $sel['from'] && $from <= $sel['to']) continue 2;
                    }
                    $to = $from;
                }
                
                # проверяем время:
                if ($this->type->placesTimeComaSeparated())
                {
                    $time = array();
                    foreach ($pv['time'] as $t) {
                        if (empty($t) || !preg_match('/[\d]{1,2}:[\d]{2}/',$t)) continue;
                        list($hour,$min) = explode(':', $t);
                        $hour = intval($hour); if ($hour>23) continue;
                        $min = intval($min); if ($min>59) continue;
                        // в пределах суток: 
                        //  - максимально ранее время события не может быть ранее 6:00 утра текущих суток
                        //  - максимально позднее время события не может быть позже 5:59 утра следующих суток
                        $time[ (($hour*60) + ($hour<=5 ? 1440 : 0) + $min) ] = sprintf('%02d:%02d', $hour, $min);
                    }
                    // if (empty($time)) continue; // время не указано
                    ksort($time);
                    $time = join(',', $time);
                } else {
                    $time = $pv['time'];
                }
                
                $placePeriods[] = array(
                    'from'     => $from,
                    'to'       => $to,
                    'from_sql' => date('Y-m-d', $from),
                    'to_sql'   => date('Y-m-d', $to),
                    'time'     => $time,
                );
            }
            
            if ( ! empty($placePeriods)) {
                $v['periods'] = $placePeriods;
                $aPlaces[] = $v;
                if ($v['place_id'] > 0) {
                    $aPlacesID[] = $v['place_id'];
                }
            }
        }
        $sqlInsert = false;

        if ( ! empty($aPlaces))
        {
            $sqlInsert = array();
            foreach ($aPlaces as $v)
            {
                foreach ($v['periods'] as $p) {
                    $sqlInsert[] = '('.$this->type->id.','.$nItemID.','.$v['place_id'].','.$this->db->str2sql($v['place_title']).','.$this->db->str2sql($p['from_sql']).','.$this->db->str2sql($p['to_sql']).','.$this->db->str2sql($p['time']).','.$nItemCityID.')';
                }
            }
            if ( ! empty($sqlInsert)) {
                $sqlInsert = 'INSERT INTO '.$sqlPlacesTable.' (type_id, item_id, place_id, place_title, period_from, period_to, period_time, city_id)
                    VALUES '.join(', ', $sqlInsert);
            } else {
                $sqlInsert = false;
            }
        }
        
        if ($bEdit)
        {
            if (empty($aPlaces)) {
                # delete current places
                $this->db->exec('DELETE FROM '.$sqlPlacesTable.' WHERE type_id = '.$this->type->id.' AND item_id = '.$nItemID);
            } else {
                # delete current places
                $this->db->exec('DELETE FROM '.$sqlPlacesTable.' WHERE type_id = '.$this->type->id.' AND item_id = '.$nItemID);
                # insert new created
                if ($sqlInsert!==false) {
                    $res = $this->db->exec($sqlInsert);
                    return !empty($res);
                }
            }
            
        } else {
            # just insert new created
            if ($sqlInsert!==false) {
                $res = $this->db->exec($sqlInsert);
                return !empty($res);
            }
        }
        
        return false;
    }
    
    function getItemPlaces($nItemID, AfishaSettings $oTypeSettings = null)
    {        
        if ($oTypeSettings === null) {
            $oTypeSettings = $this->type;
        }    

        $aPlaces = array();
        $sqlFormatDate = $this->db->str2sql('%d-%m-%Y');
        $aPeriods = $this->db->select('SELECT P.id, P.place_id, 
                        DATE_FORMAT(P.period_from,'.$sqlFormatDate.') as period_from, 
                        DATE_FORMAT(P.period_to,'.$sqlFormatDate.') as period_to, 
                        P.period_time, 
                        IF(P.place_id>0, IL.title, P.place_title) as place_title
                    FROM '.TABLE_AFISHA_PLACES.' P
                         LEFT JOIN '.TABLE_ITEMS_LANG.' IL ON P.place_id = IL.id AND IL.lang = '.$this->db->str2sql(LNG).'
                    WHERE P.item_id = '.$nItemID.' AND P.type_id = '.$oTypeSettings->id.'
                    ORDER BY 6, P.period_from');

        if ( ! empty($aPeriods)) {
            $i = 0;
            foreach ($aPeriods as $v)
            {
                $nPlaceID = ($v['place_id'] > 0 ? $v['place_id'] : ($oTypeSettings->placesMany() ? 99999 + $v['id'] : 0) );
                if ( ! isset($aPlaces[$nPlaceID])) {
                    $aPlaces[$nPlaceID] = array('id'=>($v['place_id'] > 0 ? $v['place_id'] : 0), 'title'=>$v['place_title'], 'periods'=>array());
                }
                $aPlaces[$nPlaceID]['periods'][] = array(
                    'id'   => $v['id'],
                    'from' => $v['period_from'],
                    'to'   => $v['period_to'],
                    'time' => ($oTypeSettings->placesTimeComaSeparated() ? ( ! empty($v['period_time']) ? explode(',', $v['period_time']) : array()) : $v['period_time']),
                );
            } unset($aPeriods);
        }
        return $aPlaces;
    }
    
    /**
    * Инициализация объекта работы с изображением события
    * @param array $aItemData параметры записи array(id)
    * @param AfishaSettings $oTypeSettings объект настроек типа событий
    * @return AfishaImages объект
    */
    function initImages(array $aItemData, AfishaSettings $oTypeSettings = null)
    {
        static $instance;
        if ($oTypeSettings === null) {
            $oTypeSettings = $this->type;
        }
        
        if ( ! empty($oTypeSettings) && $oTypeSettings->imgEnabled())
        {
            if (isset($instance)) {
                return $instance;
            }

            return ($instance = new AfishaImages($oTypeSettings, $aItemData));
        }
        return false;
    }

    /**
     * Инициализация компонента AfishaComments
     * @param integer $nTypeID тип
     * @return AfishaComments component
     */
    public function itemComments($nTypeID = 0)
    {
        static $c;
        if (!isset($c)) {
            $c = new AfishaComments();
            if ( ! $nTypeID) {
                $nTypeID = $this->type->id;
            }
            $c->setGroupID($nTypeID);
        }
        return $c;
    }


    /**
    * Инициализация объекта работы с видео данными события
    * @param array $aItemData параметры события array(id)
    * @param AfishaSettings $oTypeSettings объект настроек типа событий
    * @return AfishaVideo объект
    */
    function initVideo(array $aItemData, AfishaSettings $oTypeSettings = null)
    {
        static $instance;
        if ($oTypeSettings === null) {
            $oTypeSettings = $this->type;
        }
        
        if ( ! empty($oTypeSettings) && $oTypeSettings->videoEnabled())
        {
            if (isset($instance)) {
                return $instance;
            }

            return ($instance = new AfishaVideo($oTypeSettings, $aItemData));
        }
        return false;
    }
    
    /**
     * Строим путь к изображениям, исходя из типа событий
     * @param boolean $bURL стоим URL
     * @param array $aItemData данные о записи, array(id, size, filename)
     * @param AfishaSettings $oTypeSettings объект настроек типа событий
     * @return string
     */
    function getImagesPath($bURL = false, array $aItemData, AfishaSettings $oTypeSettings = null)
    {
        $bTemp = empty($aItemData['id']);
        if (empty($oTypeSettings)) {
            $oTypeSettings = $this->type;
        }

        $sPath = $this->getPath($oTypeSettings->keyword, 'images', $bURL, $bTemp);
        
        if (empty($aItemData['size'])) return $sPath;
        
        $sFilePrefix = ( $bTemp ? $oTypeSettings->id.'_' : $aItemData['id'].'_' ); 
        return $sPath . $sFilePrefix . $aItemData['size'] . $aItemData['filename'];
    }
    
    /**
     * Строим путь к видеофайлам, исходя из типа событий
     * @param boolean $bURL стоим URL
     * @param array $aItemData данные о записи, array(id, filename)
     * @param AfishaSettings $oTypeSettings объект настроек типа событий
     * @return string
     */
    function getVideoPath($bURL = false, array $aItemData, AfishaSettings $oTypeSettings = null)
    {
        $sPath = '';  
        if (empty($oTypeSettings)) {
            $oTypeSettings = $this->type;
        }

        $sPath = $this->getPath($oTypeSettings->keyword, 'video', $bURL);
        return $sPath . $aItemData['id'] . '_' . $aItemData['filename'];
    }
    
    /**
     * Строим путь к frame видеофайлов, исходя из типа событий
     * @param boolean $bURL стоим URL
     * @param array $aItemData данные о записи, array(id, filename)
     * @param AfishaSettings $oTypeSettings объект настроек типа событий
     * @return string
     */
    function getVideoFramePath($bURL = false, array $aItemData, AfishaSettings $oTypeSettings = null)
    {
        $sPath = '';  
        if (empty($oTypeSettings)) {
            $oTypeSettings = $this->type;
        }

        $sPath = $this->getPath($oTypeSettings->keyword, 'video.frame', $bURL);
        return $sPath . $aItemData['id'] . '_' . $aItemData['filename'];
    }
    
    /**
     * Строим путь к медиафайлам типа события
     * @param string $sKeyword Keyword типа
     * @param string $sContentType тип файлов: images, video, video.frame
     * @param boolean $bURL true - стоим URL, false - строим Path
     * @param boolean $bTemp временный путь
     * @return string
     */
    function getPath($sKeyword, $sContentType, $bURL = false, $bTemp = false)
    {
        switch ($sContentType)
        {
            case 'images': 
            {
                if ($bTemp) {
                    return ( $bURL ? 
                        bff::url('tmp/afisha', 'images') :
                        bff::path('tmp/afisha', 'images') );
                } else {
                    return ($bURL ? 
                        bff::url('afisha/'.$sKeyword, 'images') :
                        bff::path('afisha/'.$sKeyword, 'images') );
                }
            } break;
            case 'video':
            {
                return ($bURL ? 
                    bff::url('video/afisha/'.$sKeyword) :
                    bff::path('video/afisha/'.$sKeyword) );
            } break;
            case 'video.frame':
            {
                return ($bURL ? 
                    bff::url('video/afisha/'.$sKeyword.'/frame') :
                    bff::path('video/afisha/'.$sKeyword.'/frame') );
            } break;
        }
    }

    /**
     * Формируем заголовок записи, содержащий ссылку.
     * @param integer $nItemID ID записи
     * @param array $aItemData @ref данные о записи:
     *    title - оригинальное название, с тегами [link][/link]
     *    link - ссылка
     *    [keyword] - keyword записи
     *    category|categоry_id - keyword|ID категории записи
     *    city_id - ID города
     * @param array $aLinkAttr аттрибуты ссылки
     * @param type
     * @return boolean
     */
    public function prepareItemTitle($nItemID, array &$aItemData, $aLinkAttr = array(), $type = null)
    {
        if (!$nItemID || empty($aItemData['title'])) {
            return false;
        }

        if( ! empty($type)){
            $this->type = $type;
        }

        # сохраняем для редактирования
        $aItemData['title_params'] = serialize(array('title'=>$aItemData['title'], 'link'=>$aItemData['link']));
        $url = $this->itemURL($nItemID, $aItemData);

        # заголовок является полностью ссылкой (без выделения области ссылки)
        if (!$this->type->extra['title_link']) {
            foreach ($aItemData['title'] as $lng=>$t) {
                $aItemData['title'][$lng] = strtr($t, array('[link]'=>'','[/link]'=>''));
            }
            $aItemData['link'] = $url;
            return true;
        }

        # формируем заголовок с выделением области ссылки
        $link = $aItemData['link'];

        # включена настройка "задавать собственную ссылку"
        if ($this->type->extra['title_link_custom']) {
            if (empty($link) || $link === '#') {
                $link = $url;
            }
        } else {
            $link = $url;
        }

        # title => ссылка
        $aLinkAttr = HTML::attributes($aLinkAttr);
        foreach ($aItemData['title'] as $lng=>$t) {
            if (mb_stripos($t, '[link]') !== false) {
                $aItemData['title'][$lng] = preg_replace('|\[link\]([^\[]+)\[/link\]|iu', '<a href="'.$link.'"'.$aLinkAttr.'>$1</a>', $t);
            } else {
                # весь заголовок является ссылкой
                $aItemData['title'][$lng] = '<a href="'.$link.'"'.$aLinkAttr.'>'.$t.'</a>';
            }
        }
        $aItemData['link'] = $link;
        
        return true;
    }

    /**
     * Формируем динамический URL для записи
     * @param integer $nItemID ID записи
     * @param array $aItemData данные о записи (keyword, title, category_id, city_id)
     * @return string
     */
    protected function itemURL($nItemID, array $aItemData)
    {
        # подменяем макросы:
        # {id}
        $aReplace = array('{id}'=>$nItemID);
        # {keyword}
        if ( ! empty($aItemData['keyword'])) {
            $aReplace['{keyword}'] = $aItemData['keyword'];
        } else {
            if ( ! empty($aItemData['title'][LNG])) {
                $sKeyword = strtr($aItemData['title'][LNG], array('[link]'=>'','[/link]'=>''));
                $sKeyword = mb_strtolower(func::translit($sKeyword));
                if ( ! empty($sKeyword)) {
                    $aReplace['{keyword}'] = $sKeyword;
                }
            }
        }
        # {cat}
        if ($this->type->categoriesEnabled() && mb_stripos($this->type->linkout, '{cat}') !== false) {
            if ( ! isset($aItemData['category'])) {
                if ( ! empty($aItemData['category_id'])) {
                    $aItemData['category'] = $this->db->select_data(TABLE_AFISHA_CATEGORIES, 'keyword',
                                                array('id'=>$aItemData['category_id']));
                } else {
                    $aItemData['category'] = '';
                }
            }
            $aReplace['{cat}'] = $aItemData['category'];
        }
        $linkout = strtr($this->type->linkout, $aReplace);

        # формируем URL
        return static::urlBase(LNG, true, array('city'=>$aItemData['city_id'])).'/'.ltrim($linkout, '/');
    }
    
    function deleteItem($nItemID)
    {
        if ( ! $nItemID) return false;
        
        $nTypeID = $this->type->id;
        
        $sqlTable = $this->type->tableName();
        $aData = $this->model->itemData($nItemID, array('*'), $nTypeID);
        if (empty($aData) || !$nTypeID) return false;

        if ($this->type->contentPublicator())
        {
            # удаляем фотографии загруженные через публикатор
            $this->initPublicator($this->type)->dataDelete($aData['content'], $nItemID);
        }
        
        # удаляем изображение
        if ($this->type->imgEnabled())
        {
            $oImages = $this->initImages(array('id'=>$nItemID, 'created'=>$aData['created']), $this->type);
            if ( ! empty($aData['img']) && $oImages!==false) {
                $oImages->deleteFiles($aData['img']);
            }
        }
        
        # удаляем видео
        if ($this->type->videoEnabled())
        {
            $aVideoData = ( !empty($aData['video_data']) ?  unserialize( $aData['video_data'] ) : false );
            if ( ! empty($aVideoData) && !empty($aVideoData['file'])) {
                $oVideo = $this->initVideo( array('id'=>$nItemID), $this->type );
                $oVideo->deleteFiles( $aVideoData['file'], false );
                if ( ! empty($aVideoData['frame'])) {
                    $oVideo->deleteFiles( $aVideoData['frame'], true );
                }
            }
        }
        
        # удаляем связь с местом события
        $this->db->delete(TABLE_AFISHA_PLACES, array('type_id'=>$nTypeID, 'item_id'=>$nItemID));

        # удаляем комментарии
        if ($this->type->commentsManage()) {
            $this->db->delete(TABLE_AFISHA_COMMENTS, array('type_id'=>$nTypeID, 'item_id'=>$nItemID));
        }
        
        # актуализируем
        if ($aData['enabled']) {
            # фиксируем в календаре
            $this->calendarUpdate($nItemID, $nTypeID, 'deleted');
            # обновляем RSS
            $this->rssUpdate($nTypeID);
        }

        # удаляем привязку к тегам
        if ($this->type->tagsEnabled()) {
            $this->initTags($nTypeID)->onItemDelete($nItemID);
        }

        # удаляем мультиязычные данные
        $this->db->delete(TABLE_AFISHA_ITEMS_LANG, array('type_id'=>$nTypeID, 'id'=>$nItemID));

        # удаляем запись
        $res = $this->db->delete($sqlTable, array('id'=>$nItemID));

        # корректируем счетчик модерации
        $deleted = !empty($res);
        if ($deleted && $this->type->moderationEnabled() && !$aData['moderated']) {
            $this->itemsModerationCounter($this->type->keyword, -1);
        }
        
        return $deleted;
    }

    function itemsModerationCounter($sKey, $nIncrement = false)
    {
        $sCounterKey = 'afisha_items_mod_'.$sKey;
        if ($nIncrement!==false) {
            config::saveCount($sCounterKey, $nIncrement, true);
        } else {
            return (int)config::get($sCounterKey, 0);
        }
    }

    function itemViewsCounter($nItemID)
    {
        return $this->db->exec('UPDATE '.$this->type->tableName().' SET views = views + 1 WHERE id = '.$nItemID);
    }
    
    /**
     * Инициализация компонента Publicator
     * @param AfishaSettings $oTypeSettings объект настроек типа событий
     * @return Publicator объект
     */
    function initPublicator(AfishaSettings $oTypeSettings = null)
    {
        if ($oTypeSettings === null) {
            $oTypeSettings = $this->type;
        }
        if (empty($oTypeSettings->content_pub['sett'])) {
            $this->errors->set(_t('', 'Ошибка инициализация компонента Publicator, проверьте корректность настроек рубрики'));
            return;
        }

        // формируем всегда конечные пути хранения изображений
        // на их основе публикатор сформирует временные, путем добавления "/tmp"
        $aItemData = array('id'=>1000);

        $aSettings = $oTypeSettings->content_pub['sett'];
        $aSettings['images_path'] = $this->getImagesPath(false, $aItemData, $oTypeSettings);
        $aSettings['images_path_tmp'] = bff::path('tmp', 'images');
        $aSettings['images_url']  = $this->getImagesPath(true, $aItemData, $oTypeSettings);
        $aSettings['images_url_tmp'] = bff::url('tmp', 'images');
        $aSettings['langs'] = $this->locale->getLanguages(true);

        return $this->attachComponent('publicator', new Publicator($this->module_name, $aSettings, true));
    }

    function updateContentType($aPublicatorData)
    {   
        $nContentTypes = 0;
        if ($this->type->contentTypesEnabled() &&
           ! empty($aPublicatorData['content_types']) &&
           array_sum($aPublicatorData['content_types'])>0) 
        {
            foreach ($aPublicatorData['content_types'] as $type=>$typeBlocks) {
                if ( ! $typeBlocks) continue;
                if ($type == Publicator::blockTypePhoto || $type == Publicator::blockTypeGallery) {
                    $nContentTypes |= self::contentTypePhoto;
                } else if ($type == Publicator::blockTypeVideo) {
                    $nContentTypes |= self::contentTypeVideo;
                }
            }
        }
        return $nContentTypes;
    }

    /**
     * Инициализация компонента работы с дин. свойствами
     * @return \bff\db\Dynprops объект
     */
    function dp()
    {
        static $oDp = null;
        if (isset($oDp)) return $oDp;

        $oDp = $this->attachComponent('dynprops', new Dynprops('owner_id',
                TABLE_AFISHA_TYPES, TABLE_AFISHA_DYNPROPS, TABLE_AFISHA_DYNPROPS_MULTI) );
                
        $oDp->setSettings(array(
            'module_name'=>'afisha',
            'cache_method'=>'Afisha_dpSettingsUpdated',
            'datafield_int_first' => 1,
            'datafield_int_last' => 2,
            'datafield_text_first' => 3,
            'datafield_text_last' => 10,
            'typesAllowed'=>array(
                Dynprops::typeCheckboxGroup,
                Dynprops::typeRadioGroup,
                Dynprops::typeRadioYesNo,
                Dynprops::typeCheckbox,
                Dynprops::typeSelect,
                Dynprops::typeSelectMulti,
                Dynprops::typeInputText,
                Dynprops::typeTextarea,
                Dynprops::typeNumber,
                Dynprops::typeRange,
            ),
            'typesAllowedParent'=>array(),
            'langs' => $this->locale->getLanguages(false),
            'ownerTable_Title' => 'title_'.LNG,
        ));
    
        return $oDp;
    }

    function dpSettingsUpdated($nTypeID, $nDynpropID, $sEvent)
    {
        if (empty($nTypeID)) return false;
        # сбрасываем кеш настроек типа
        $this->resetTypeSettingsCache($nTypeID, false);
    }
    
    function dpPrepareSelectFieldsQuery($sPrefix = '', AfishaSettings $oTypeSettings = null)
    {
        if ($oTypeSettings===null) {
            $oTypeSettings = $this->type;
        }
        if ($oTypeSettings===null)
            return '';
        
        $fields = array();
        foreach ($oTypeSettings->dp as $v)
        {
            $f = $sPrefix.'f'.$v['data_field'];
            if ( ! empty($v['cache_key'])) {
               $f .= ' as '.$v['cache_key'];
            }
             $fields[] = $f;
        }
        return ( ! empty($fields) ? ', ' . join(', ', $fields) : '');
    }
    
    function dpGetMainGenreTitle($nGenreID = 0)
    {
        static $cache, $id = 15;
        if ( ! isset($cache))
        {
            $data = $this->dp()->getByID(array($id), true);
            $cache = array();
            if ( ! empty($data[$id]))
            {
                foreach ($data[$id]['multi'] as $m) {
                    $cache[$m['value']] = $m['name'];
                }
            }
        }
        return (isset($cache[$nGenreID]) ? $cache[$nGenreID] : '');
    }
    
    /**
     * Ставим хук на вызов неизвестного метода, 
     * для работы с данными типа событий:
     * @param string [component|module]_[method] или [type-keyword]:[method]
     * @param array $aArgs
     * @return mixed
     */
    public function __call($sName, $aArgs = array()) 
    {
        if (strpos($sName, ':')!==false)
        {
            list($sTypeKey, $sMethod) = explode(':', $sName, 2);
            $nTypeID = $this->getTypeIDByKeyword($sTypeKey);
            
            if ($nTypeID!==false && method_exists($this, $sMethod))
            {
                # инициализируем настройки текущего типа
                $this->type = $this->getTypeSettings($nTypeID);
                $_GET['type'] = $_POST['type'] = $this->type->id; 
                if ( ! bff::adminPanel() ) {
                    $this->typeTemplatesDir = $this->module_dir_tpl . DS . $sTypeKey;
                    $this->itemsTable = $this->type->tableName();
                }
                if ($this->type->contentPublicator()) {
                    unset($this->model->langItems['content']);
                }

                return $this->$sMethod($this->type->id, $sTypeKey);
            } else {
                if (!bff::adminPanel()) {
                    $this->errors->error404();
                }
            }
        }

        return parent::__call($sName, $aArgs); 
    }
    
    /**
     * Построение меню админ панели для работы с модулем афиша
     */
    function buildAdminMenu(CMenu $menu)
    {
        $aTypes = $this->getTypes();
        $i = 20;
        $sModule = 'afisha';

        $access_Comments = $this->haveAccessTo('comments', $sModule);
        $sModulePrefix = $sModule.'-';
        
        if ($this->menuMini)
        {
            $tagsEnabled = false;
            foreach ($aTypes as $sKey=>$v) {
                if ( ! $this->haveAccessTo('items-listing', $sModulePrefix.$sKey)) continue;

                $params = array('rlink'=>array('event'=>$sKey.':add'));
                if ($v->moderationEnabled()) {
                    $params['counter'] = 'afisha_items_mod_'.$sKey;
                }    
                $menu->assign('Афиша', $v->title, $sModule, $sKey.':listing', true, $i++, $params);
                $menu->assign('Афиша', $v->title.' / Добавление', $sModule, $sKey.':add', false, $i++);
                $menu->assign('Афиша', $v->title.' / Редактирование', $sModule, $sKey.':edit', false, $i++);
                if ($v->tagsEnabled()) {
                    $tagsEnabled = true;
                }
            }

            if ($access_Comments)
            {
                $aCommentsParams = array();
                if (bff::$event!=='comments_mod') {
                    $aCommentsParams['counter'] = 'afisha_comments_mod_all';
                }
                $menu->assign('Афиша', 'Комментарии', $sModule, 'comments_mod', true, $i++, $aCommentsParams);
            }
//            if ($this->haveAccessTo('cats-listing'))
//                $menu->assign('Афиша', 'Разделы', $sModule, 'categories', true, $i++);
            if ($tagsEnabled && $this->haveAccessTo('tags-listing')) {
                $menu->assign('Афиша', 'Теги', $sModule, 'tags', true, $i++);
            }
            
        } else {
            foreach ($aTypes as $sKey=>$v)
            {
                if ( ! $this->haveAccessTo('items-listing', $sModulePrefix.$sKey)) continue;
                
                $menu->assign($v->title, 'Список', $sModule, $sKey.':listing', true, $i++, array('rlink'=>array('event'=>$sKey.':add')));
                $menu->assign($v->title, 'Добавление', $sModule, $sKey.':add', false, $i++);
                $menu->assign($v->title, 'Редактирование', $sModule, $sKey.':edit', false, $i++);
                if ( $v->categoriesEnabled() && $this->haveAccessTo('cats-listing', ( ! $v->categoriesCommonOnly() ? $sModulePrefix.$sKey : $sModule)) ) {
                    $menu->assign($v->title, 'Разделы', $sModule, $sKey.':categories', true, $i++);
                }
                if ( $v->tagsEnabled() && $this->haveAccessTo('tags-listing')) {
                    $menu->assign($v->title, 'Теги', $sModule, $sKey.':tags', true, $i++);
                }
                if ( $access_Comments && $v->commentsManage() ) {
                    $menu->assign($v->title, 'Комментарии', $sModule, $sKey.':comments_mod', true, $i++);
                }
            }
        }

        if (FORDEV)
        {
            $menu->assign('Афиша', 'Настройки рубрик', $sModule, 'types_listing', true, $i++, array(
                'rlink'=>array('event'=>'types_add')
            ));
            $menu->assign('Афиша', 'Добавление рубрики', $sModule, 'types_add', false, $i++);
            $menu->assign('Афиша', 'Редактирование рубрики', $sModule, 'types_edit', false, $i++);
        }
    }
    
    /**
     * Обработка данных типа событий 
     * @param array $aData @ref настройки типа событий
     * @param mixed $nTypeID ID типа
     * @return void
     */
    function processTypeSettings(&$aData, $nTypeID = 0)
    {
        $aParams = array(
            'keyword' => TYPE_NOTAGS,
            'regions' => TYPE_UINT,
            'regions_country' => TYPE_UINT,
            'categories'      => TYPE_UINT,
            'tags'            => TYPE_UINT, 
            'comments'        => TYPE_UINT, 
            'commentsp'       => TYPE_ARRAY,
            'author'        => TYPE_UINT,
            'img'           => TYPE_BOOL,
            'imgp'          => TYPE_ARRAY,
            'video'         => TYPE_UINT,
            'content_short' => TYPE_BOOL,
            'content'       => TYPE_UINT,
            'content_pub'   => TYPE_ARRAY,
            'content_types' => TYPE_UINT,
            'places'        => TYPE_ARRAY,
            'users_add'     => TYPE_UINT,
            'users_groups'  => TYPE_ARRAY_UINT,
            'extra'         => TYPE_ARRAY,
            'linkout'       => TYPE_STR,
            'enabled'       => TYPE_BOOL,
        );
        $this->input->postm($aParams, $aData);
        $this->input->postm_lang($this->model->langTypes, $aData );


        $aData['imgp']        = AfishaSettings::cleanArray($aData['imgp'], 'imgp');
        $aData['commentsp']   = AfishaSettings::cleanArray($aData['commentsp'], 'commentsp');
        $aData['content_pub'] = AfishaSettings::cleanArray($aData['content_pub'], 'content_pub');
        $aData['places']      = AfishaSettings::cleanArray($aData['places'], 'places');
        $aData['extra']       = AfishaSettings::cleanArray($aData['extra'], 'extra');
        
        if (Request::isPOST())
        {
            $aData['keyword'] = $this->getKeyword($aData['keyword'], $aData['title'][LNG], TABLE_AFISHA_TYPES, $nTypeID);
            
            //изображение 
            {
                $imgp =& $aData['imgp'];
                
                if ($imgp['orig'] && $imgp['size_orig']=='') {
                    $imgp['orig'] = false;
                }
                
                # размеры изображений
                $imgSizes = array(); $imgPrefixes = array(); $cropSizes = array(); $wmSizes = array();
                if ( ! empty($imgp['sizes']))
                {
                    foreach ($imgp['sizes'] as $v)
                    {
                        $this->input->clean_array($v, array(
                            'p' => TYPE_STR,     // префикс
                            'w' => TYPE_UINT,    // ширина
                            'h' => TYPE_UINT,    // высота
                            'wm' => TYPE_BOOL,   // watermark
                            'crop' => TYPE_BOOL, // crop
                            'del' => TYPE_BOOL,  // пометка на удаление размера
                        ));
                        
                        $prefix = mb_strtolower($v['p']);
                        $prefix = preg_replace('/[^a-z]/','', $prefix);
                        if ((empty($v['p']) || $v['p']==$imgp['size_orig'] || in_array($prefix, $imgPrefixes)) || ( ! $v['w'] && !$v['h'])) continue;
                        
                        if ($nTypeID>0 && $v['del']) {
                            //удаление размера
                            continue;
                        } else {
                            unset($v['del']);
                        }
                        if ( ! $v['w']) $v['w'] = false;
                        if ( ! $v['h']) $v['h'] = false;
                        if ($v['crop']) $cropSizes[] = $prefix;
                        if ($v['wm']) $wmSizes[] = $prefix;
                        $imgPrefixes[] = $prefix;
                        
                        $v['p'] = $prefix;
                        $imgSizes[$prefix] = $v; 
                    }
                }
                $imgp['sizes'] = $imgSizes;  
                
                if ( ! array_key_exists($imgp['size_adm'], $imgSizes)) {
                     $imgp['size_adm'] = key($imgSizes);
                }
                
                # watermark
                $wmValid = false;
                do {
                    if ( ! $imgp['wm'] || !in_array($imgp['wm'], array(1,2)) || empty($imgSizes)) break;
                    
                    $wmError = false;
                    if ($imgp['wm'] == 1) { // изображение
                        if ( ! file_exists(PATH_PUBLIC . $imgp['wm_img'])) {
                            $this->errors->set( _t('afisha', 'Путь watermark-изображения "[path]" указан некорректно', array('path'=>$imgp['wm_img'])), true );
                            $wmError = true;
                        }
                        if (empty($imgp['wm_pos_x']) || !in_array($imgp['wm_pos_x'], array('left','center','right')))
                            $imgp['wm_pos_x'] = 'right';
                        if (empty($imgp['wm_pos_y']) || !in_array($imgp['wm_pos_y'], array('top','center','bottom')))
                            $imgp['wm_pos_y'] = 'bottom';
                    } else if ($imgp['wm'] == 2) { // текст
                        if ($imgp['wm_text'] == '') {
                            $this->errors->set( _t('afisha', 'Текст для watermark указан некорректно') );
                            $wmError = true;
                        }
                        if (empty($imgp['wm_pos_text_x']) || !in_array($imgp['wm_pos_text_x'], array('left','center','right')))
                            $imgp['wm_pos_text_x'] = 'right';
                        if (empty($imgp['wm_pos_text_y']) || !in_array($imgp['wm_pos_text_y'], array('top','center','bottom')))
                            $imgp['wm_pos_text_y'] = 'bottom';
                        
                        $imgp['wm_pos_x'] = $imgp['wm_pos_text_x'];
                        $imgp['wm_pos_y'] = $imgp['wm_pos_text_y'];
                    }
                    
                    if ($wmError || empty($wmSizes)) {
                        break; // размеры для wm не указаны
                    }
                    
                    $wmValid = true;
                } while(false);
                $imgp['wm'] = ($wmValid ? 1 : 0);

                # crop
                $imgp['crop'] = ( $imgp['orig'] && !empty($imgSizes) && !empty($cropSizes) ? $cropSizes : false );
            }
            
            //publicator
            {
                $pub =& $aData['content_pub'];

                $pub['title'] = (bool)$pub['title'];
                $pub['use_wysiwyg'] = (bool)$pub['use_wysiwyg'];
                $pub['photo_wysiwyg'] = (bool)$pub['photo_wysiwyg'];

                $sett = array();
                // настройки сразу подходящие публикатору:
                foreach (array('title','use_wysiwyg','images_original',
                        'photo_wysiwyg','gallery_photos_limit',
                        'video_width','video_height') as $k) {
                    $sett[$k] = $pub[$k];
                }
                $sett['controls'] = array(Publicator::blockTypeText, Publicator::blockTypeSubtitle);

                // приводим к формату, понимаемому публикатором:
                # Photo
                if ( $pub['photo_enabled'] )
                {
                    $sett['photo_sz_view'] = array(
                        'width' => $pub['photo_view_width'],
                        'height' => $pub['photo_view_height'],
                    );
                    do {
                        if ( ! $pub['photo_view_wm']) break;

                        if ( ! file_exists(PATH_PUBLIC . $pub['photo_view_wm_src'])) {
                            $this->errors->set( _t('afisha', 'Путь watermark-изображения "[path]" для Фото публикатора указан некорректно', array('path'=>$pub['photo_view_wm_src'])), true );
                        } else {
                            $sett['photo_sz_view'] = array_merge($sett['photo_sz_view'], array(
                                'watermark'=>true, 'watermark_src'=>PATH_PUBLIC . $pub['photo_view_wm_src'], 'watermark_resizable'=>false,
                                'watermark_pos_x'=>$pub['photo_view_wm_pos_x'], 'watermark_pos_y'=>$pub['photo_view_wm_pos_y']));
                        }
                    } while(false);
                    $sett['controls'][] = Publicator::blockTypePhoto;
                }
                # Gallery
                if ( $pub['gallery_enabled'] )
                {
                    $sett['gallery_sz_view'] = array(
                        'width' => $pub['gallery_view_width'],
                        'height' => $pub['gallery_view_height'],
                    );
                    do {
                        if ( ! $pub['gallery_view_wm']) break;

                        if ( ! file_exists(PATH_PUBLIC . $pub['gallery_view_wm_src'])) {
                            $this->errors->set( _t('afisha', 'Путь watermark-изображения "[path]" для Фотогалереи публикатора указан некорректно', array('path'=>$pub['gallery_view_wm_src'])), true );
                        } else {
                            $sett['gallery_sz_view'] = array_merge($sett['gallery_sz_view'], array(
                                'watermark'=>true, 'watermark_src'=>PATH_PUBLIC . $pub['gallery_view_wm_src'], 'watermark_resizable'=>false,
                                'watermark_pos_x'=>$pub['gallery_view_wm_pos_x'], 'watermark_pos_y'=>$pub['gallery_view_wm_pos_y']));
                        }
                    } while(false);
                    $sett['controls'][] = Publicator::blockTypeGallery;
                }
                # Video
                if ( $pub['video_enabled'] )
                {
                    $sett['controls'][] = Publicator::blockTypeVideo;
                }
                
                $pub['sett'] = $sett;
            }

            //подкатегории справочника
            foreach ($aData['places']['categories'] as $k=>$v) {
                if ( ! $v) { // удаляем id категорий == 0
                    unset($aData['places']['categories'][$k]); 
                }
            }
            
            # "города" => но, не указана страна
            if ($aData['regions'] == 1 && empty($aData['regions_country']))
                $aData['regions'] = 0;
                
            if ($aData['users_add'])
            {
                $aGroups = $aData['users_groups'];
                $aFrontendGroups = Users::model()->groups(null, true);
                $aFrontendGroups = func::array_transparent($aFrontendGroups, 'group_id', true);
                foreach ($aGroups as $k=>$v) {
                    if ( ! array_key_exists($v, $aFrontendGroups))
                        unset($aGroups[$k]);
                }
                $aData['users_groups'] = $aGroups;
            }
            
            if (empty($aData['linkout'])) {
                $aData['linkout'] = "/{$aData['keyword']}/{id}.html";
            }
        }
    }
    
    function getTypeSettingsForm($sKey, $mCurrent, $bHTML = true)
    {
        $sHTML = '';
        switch ($sKey)
        {
            case 'regions': 
            {
                $aValues = array(0=>'нет', 1=>'города', 2=>'страны и города');
                if ($bHTML) {
                    foreach ($aValues as $k=>$v) {
                        $sHTML .= '<option value="'.$k.'"'.($mCurrent==$k?' selected':'').'>'.$v.'</option>';
                    }
                } else {
                    return $aValues;
                }
            } break;
            case 'regions_country': 
            {
                $aCountries = Geo::countryList();
                if ($bHTML) {
                    $sHTML = HTML::selectOptions($aCountries, $mCurrent, '- выбрать страну -', 'id', 'title');
                } else {
                    return $aCountries;
                }
            } break;
            case 'categories':
            {
                $aValues = array(0=>'нет', 1=>'свои', 2=>'свои + общие (выборочное наследование)', 3=>'только общие (полное наследование)');
                if ($bHTML) {
                    foreach ($aValues as $k=>$v) {
                        $sHTML .= '<option value="'.$k.'"'.($mCurrent==$k?' selected':'').'>'.$v.'</option>';
                    }
                } else {
                    return $aValues;
                }
            } break;
            case 'tags':
            {
                $aValues = array(0=>'нет', 1=>'свои', 2=>'общие');
                if ($bHTML) {
                    foreach ($aValues as $k=>$v) {
                        $sHTML .= '<option value="'.$k.'"'.($mCurrent==$k?' selected':'').'>'.$v.'</option>';
                    }
                } else {
                    return $aValues;
                }
            } break;  
            case 'comments':
            {
                $aValues = array(0=>'нет', 1=>'древовидные', 2=>'недревовидные', 3=>'facebook', 4=>'vkontakte');
                if ($bHTML) {
                    foreach ($aValues as $k=>$v) {
                        $sHTML .= '<option value="'.$k.'"'.($mCurrent==$k?' selected':'').'>'.$v.'</option>';
                    }
                } else {
                    return $aValues;
                }
            } break;  
            case 'comments_mod':
            {
                $aValues = array(1=>'премодерация', 0=>'постмодерация');
                if ($bHTML) {
                    foreach ($aValues as $k=>$v) {
                        $sHTML .= '<option value="'.$k.'"'.($mCurrent==$k?' selected':'').'>'.$v.'</option>';
                    }
                } else {
                    return $aValues;
                }
            } break;
            case 'content': 
            {
                $aValues = array(2=>'textarea', 0=>'wysiwyg', 1=>'publicator');
                if ($bHTML) {
                    foreach ($aValues as $k=>$v) {
                        $sHTML .= '<option value="'.$k.'"'.($mCurrent==$k?' selected':'').'>'.$v.'</option>';
                    }
                } else {
                    return $aValues;
                }
            } break;
            case 'video': 
            {                 
                $aValues = array(0=>'нет', 1=>'код или файл');
                if ($bHTML) {
                    foreach ($aValues as $k=>$v) {
                        $sHTML .= '<option value="'.$k.'"'.($mCurrent==$k?' selected':'').'>'.$v.'</option>';
                    }
                } else {
                    return $aValues;
                }
            } break; 
            case 'places_time': 
            {
                $aValues = array(0=>'нет', 1=>'перечисление', 2=>'текстовое поле');
                if ($bHTML) {
                    foreach ($aValues as $k=>$v) {
                        $sHTML .= '<option value="'.$k.'"'.($mCurrent==$k?' selected':'').'>'.$v.'</option>';
                    }
                } else {
                    return $aValues;
                }
            } break; 
            case 'author':
            {
                $aValues = array(0=>'нет', 1=>'текст', 2=>'пользователь');
                if ($bHTML) {
                    foreach ($aValues as $k=>$v) {
                        $sHTML .= '<option value="'.$k.'"'.($mCurrent==$k?' selected':'').'>'.$v.'</option>';
                    }
                } else {
                    return $aValues;
                }
            } break;
            case 'users_add':
            {
                $aValues = array(0=>'нет', 1=>'премодерация', 2=>'постмодерация', 3=>'без модерации');
                if ($bHTML) {
                    foreach ($aValues as $k=>$v) {
                        $sHTML .= '<option value="'.$k.'"'.($mCurrent==$k?' selected':'').'>'.$v.'</option>';
                    }
                } else {
                    return $aValues;
                }
            } break;
            case 'users_groups': 
            {
                $aFrontendGroups = Users::model()->groups(null, true);
                if ($bHTML) {
                    foreach ($aFrontendGroups as $k=>$v) {
                        $sHTML .= '<option value="'.$v['group_id'].'" style="color:'.$v['color'].';" '.(in_array($v['group_id'], $mCurrent)?' selected':'').'>'.$v['title'].'</option>';
                    }
                } else {
                    return $aFrontendGroups;
                }
            } break;
        }
        
        return $sHTML;
    }
    
    /**
     * @return AfishaSettings|array
     */
    public function getTypeSettings($nTypeID, $bIgnoreCache = false, $bReturnArray = false)
    {
        static $cache = array();
        if ( ! $bIgnoreCache && isset($cache[$nTypeID])) {
            return $cache[$nTypeID];
        }

        $cKey = 'type-sett-'.$nTypeID;
        $c = $this->initTypeSettingsCache(); 
        if ( $bIgnoreCache || ($aSettings = $c->get($cKey)) === false ) // ищем в кеше
        {
            $aSettings = $this->db->one_array('SELECT * FROM '.TABLE_AFISHA_TYPES.' WHERE id = :id', array(':id'=>$nTypeID));
            $this->db->langFieldsSelect($aSettings, $this->model->langTypes);

            AfishaSettings::unpack($aSettings);
            if ( ! $bIgnoreCache) {
                $aSettings['dp'] = $this->dp()->getByOwner($nTypeID, false, true, false);
                $c->set($cKey, $aSettings); // сохраняем в кеш
            }
        }

        $cache[$nTypeID] = new AfishaSettings($aSettings, false);
        return ( $bReturnArray ? $aSettings : $cache[$nTypeID] );
    }
    
    public function getTypeIDByKeyword($sKeyword)
    {
        $sKeyword = strtoupper($sKeyword);
        if (defined('self::TYPE_'.$sKeyword)) {
            return constant('self::TYPE_'.$sKeyword);
        }
        return false;
    }

    function getTypeKeywordByID($typeID)
    {
        return $this->getTypeSettings($typeID)->keyword;
    }

    function initTypeSettingsCache()
    {
        return Cache::singleton('afisha', 'file');
    }
    
    function resetTypeSettingsCache($nTypeID = 0, $bAll = false)
    {
        # сбрасываем кеш настроек типа
        $cache = $this->initTypeSettingsCache();
        if ($nTypeID > 0) {
            $cache->delete('type-sett-'.$nTypeID);
        }
        if ($bAll) {
            $cache->delete('type-sett-all');
        }
    }
    
    /**
     * Получение настроек всех типов одним массивом
     * Для использования в админ. панели
     * ! Без кеша динамических свойств
     * @return array (key=>AfishaSettings, key=>...)
     */
    public function getTypes()
    {
        # скрываем вылюченные, если не в режиме FORDEV
        $sqlQuery = 'SELECT *, title_'.LNG.' AS title FROM '.TABLE_AFISHA_TYPES.' '.( !FORDEV ?' WHERE enabled = 1 ':'').' ORDER BY num';
        $c = $this->initTypeSettingsCache();
        $cacheKey = 'type-sett-all';
        
        if ( ! FORDEV) {
            if ( ( $aTypes = $c->get($cacheKey) ) === false ) { // берем из кеша
                $aTypes = $this->db->select( $sqlQuery );
                $aTypes = func::array_transparent($aTypes, 'keyword', true);
                foreach ($aTypes as $key=>$sett) {
                    $aTypes[$key] = new AfishaSettings( $sett );
                }
                $c->set($cacheKey, $aTypes); // сохраняем в кеш
            }
            return $aTypes;
        }
        $aTypes = $this->db->select( $sqlQuery );
        if ( ! empty($aTypes)) {
            $aTypes = func::array_transparent($aTypes, 'keyword', true);
            foreach ($aTypes as $key=>$sett) {
                $aTypes[$key] = new AfishaSettings( $sett );
            }
        }
        $c->delete($cacheKey); // очищаем кеш в debug-режиме
        return $aTypes;
    }
    
    function isKeywordExists($sKeyword, $sTable, $nExceptRecordID = null) //ss
    {
        return $this->db->one_data('SELECT id
                               FROM '.$sTable.'
                               WHERE '.( !empty($nExceptRecordID)? ' id!='.intval($nExceptRecordID).' AND ' : '' ).'
                                  keyword='.$this->db->str2sql($sKeyword).'  LIMIT 1');
    }
    
    function getKeyword($sKeyword = '', $sTitle = '', $sTable, $nExceptRecordID = null)
    {
        if (empty($sKeyword) && ! empty($sTitle)) {
            $sKeyword = mb_strtolower( func::translit( $sTitle ) );
        }
        $sKeyword = preg_replace('/[^a-zA-Z0-9]/','',$sKeyword);
        
        if (empty($sKeyword)) {
             $this->errors->set(_t('', 'Укажите keyword'));
        } else {
            if ( $this->isKeywordExists($sKeyword, $sTable, $nExceptRecordID) )
             $this->errors->set(_t('','Указанный keyword уже используется'));

            if ( ! $nExceptRecordID) {
                $sImagesPath = $this->getPath($sKeyword, 'images');
                if (is_dir($sImagesPath) ||
                   $this->db->isTable(TABLE_AFISHA_ITEMS_.$sKeyword) ) 
                {
                    $this->errors->set(_t('','Указанный keyword уже используется'));
                }
            }
        }
        return $sKeyword;
    }
    
    function deleteType($nTypeID)
    {
        if ($nTypeID<=0 || ! FORDEV) return false;
        
        $type = $this->getTypeSettings($nTypeID, true);
        if (empty($type)) return false;
        
        $key = $type->keyword;
        
        $this->manageTypeFiles($key, 'delete');
        
        //категории: свои или свои + общие(частично)
        if ($type->categoriesCustomOnly() || $type->categoriesCustomAndCommon())
        {
            $this->db->exec('
                DELETE CL, C
                FROM '.TABLE_AFISHA_CATEGORIES.' C,
                     '.TABLE_AFISHA_CATEGORIES_LANG.' CL
                WHERE C.id = CL.id AND C.type_id = '.$nTypeID);
            $this->db->exec('DELETE FROM '.TABLE_AFISHA_CATEGORIES_IN.' WHERE type_id = '.$nTypeID);
        }
        
        //теги: свои
        if ($type->tagsCustom()) {
            $this->db->exec('DELETE FROM '.TABLE_AFISHA_TAGS.' WHERE type_id = '.$nTypeID);
        }

        //чистим комментарии
        $this->db->exec('DELETE FROM '.TABLE_AFISHA_COMMENTS.' WHERE type_id = '.$nTypeID);
        
        //чистим места
        $this->db->exec('DELETE FROM '.TABLE_AFISHA_PLACES.' WHERE type_id = '.$nTypeID);
        
        //удаляем таблицу записей
        $this->db->exec('DROP TABLE IF EXISTS '. $type->tableName() );

        $this->db->exec('DELETE FROM '.TABLE_AFISHA_ITEMS_LANG.' WHERE type_id = '.$nTypeID);

        $this->db->exec('DELETE FROM '.TABLE_AFISHA_TYPES.' WHERE id = '.$nTypeID);
    }
    
    function manageTypeFiles($sKeyword, $sAction)
    {
        switch ($sAction)
        {
            case 'create': 
            case 'delete':
            {
                if (empty($sKeyword)) {
                    $this->errors->set( _t('afisha', 'Keyword типа указан некорректно') );
                    return false;
                }
                
                $aFolders = array(
                    'tpl'   => $this->module_dir_tpl.DS.$sKeyword.DS,
                    'img'   => $this->getPath($sKeyword, 'images'),
                    'video' => $this->getPath($sKeyword, 'video'),
                );
                
                if ($sAction == 'create') {
                    # создаем директории:
                    //$nChmod = 0666;
                    $nChmod = null;
                    mkdir( $aFolders['tpl'], $nChmod); # для шаблонов
                    mkdir( $aFolders['img'], $nChmod); # для изображений
                    mkdir( $aFolders['video'], $nChmod); # для video
                } else {
                    # удаляем директории:
                    rmdir( $aFolders['tpl'] ); # для шаблонов 
                    rmdir( $aFolders['img'] ); # для изображений
                    rmdir( $aFolders['video'] ); # для video
                }
            } break;
            case 'rename': 
            {
                if ( ! empty($sKeyword['from']) && !empty($sKeyword['to']))
                {
                    # переименовываем директории:
                    rename( $this->module_dir_tpl.DS.$sKeyword['from'].DS,
                            $this->module_dir_tpl.DS.$sKeyword['to'].DS );
                    rename( $this->getPath($sKeyword['from'], 'images'),
                            $this->getPath($sKeyword['to'],   'images') );
                    rename( $this->getPath($sKeyword['from'], 'video'), 
                            $this->getPath($sKeyword['to'],   'video') );
                }
            } break;
        }
    }

    function manageTypeTables($sKeyword, $sAction)
    {
        if (empty($sKeyword)) {
            $this->errors->set( _t('afisha', 'Keyword типа указан некорректно') );
            return false;
        }
        
        switch ($sAction)
        {
            case 'create':
            {
                $sqlInstall = Files::getFileContent($this->module_dir.'install.type.sql');
                $sqlInstall = strtr($sqlInstall, array('{key}'=>$sKeyword));
                $res = $this->db->exec($sqlInstall);
                return ($res===false ? false : true);
            } break;
            case 'rename': 
            {
                if ( ! empty($sKeyword['from']) && !empty($sKeyword['to']))
                {
                    if ( $this->db->isTable(TABLE_AFISHA_ITEMS_.$sKeyword['from']) &&
                      ! $this->db->isTable(TABLE_AFISHA_ITEMS_.$sKeyword['to']) )
                    {
                        $res = $this->db->exec('RENAME TABLE '.TABLE_AFISHA_ITEMS_.$sKeyword['from'].' TO '.TABLE_AFISHA_ITEMS_.$sKeyword['to'].'');
                        return ($res===false ? false : true);
                    }
                    return false;
                }
            } break;
        }
    }
    
    # категории
    
    function validateCategoryData($nTypeID = 0)
    {
        $aParams = array(
            'keyword'      => TYPE_NOTAGS, # keyword
            'enabled'      => TYPE_BOOL,   # включена ли категория
        );
        $aData = $this->input->postm($aParams);
        $this->input->postm_lang($this->model->langCategories, $aData );

        if (Request::isPOST())
        {
            $aData['type_id'] = $nTypeID;
            if ($aData['title'] == '') {
                $this->errors->set( _t('afisha', 'Название категории указано некорректно') );
            }
            if ($aData['keyword'] == '') {
                $this->errors->set( _t('afisha', 'Keyword категории указан некорректно') );
            }            
        }
        return $aData;
    }
    
    function categoriesGetCurrentNum($nTypeID = 0, $bInheritItable = null)
    {
        $nNum = $this->db->one_data('SELECT MAX(num) FROM '.($bInheritItable ? TABLE_AFISHA_CATEGORIES_IN : TABLE_AFISHA_CATEGORIES).' 
                                     WHERE type_id = '.$nTypeID);
        return intval($nNum) + 1;
    }

    /**
    * @param AfishaSettings|null $oTypeSettings объект настроек типа событий
    * @param boolean $bEnabled только влюченные категории
    * @param mixed $mOptions false - возвращать массив категорий, array(sel, empty) - возвращать html <options>
    * @param boolean $bCountItems подсчитывать ли кол-во записей в категориях
    * @return mixed (string | array)
    */
    function categoriesGet(AfishaSettings $oTypeSettings = null, $bEnabled = true, $mOptions = false, $bCountItems = false)
    {
        $bGetOptions = !empty($mOptions);
        $bCountItems = ($bCountItems && !empty($oTypeSettings) && !$bGetOptions);

        $sqlWhere = array();
        $sqlSelect = array();

        if (empty($oTypeSettings)) {
            $oTypeSettings = new AfishaSettings( array('categories'=>3), false ); //общие категории \/
        }
        switch ( $oTypeSettings->categories )
        {
            case 0: //нет
            {
                return ( $bGetOptions ? '<option value="0">'._t('', 'нет доступных разделов').'</option>' : array());
            } break;
            case 1: //свои
            case 3: //только общие (полное наследование)
            {
                if ($bGetOptions) $sqlSelect = array('C.id','CL.title');
                else {
                    $sqlSelect = array('C.*','CL.*');
                    if ($bCountItems) $sqlSelect[] = 'COUNT(I.id) as items';
                }        
                
                $sqlSelect = join(',', $sqlSelect);

                $sqlWhere[] = 'C.type_id = '.($oTypeSettings->categoriesCustomOnly() ? $oTypeSettings->id : 0);
                if ($bEnabled) $sqlWhere[] = 'C.enabled = 1';
                $aCategories = $this->db->select('SELECT '.$sqlSelect.', 0 as inherited
                        FROM '.TABLE_AFISHA_CATEGORIES.' C
                            '.($bCountItems ? ' LEFT JOIN '.$oTypeSettings->tableName().' I ON I.category_id = C.id ' : '').',
                            '.TABLE_AFISHA_CATEGORIES_LANG.' CL
                        WHERE '.join(' AND ', $sqlWhere).$this->db->langAnd(true, 'C', 'CL').'
                        '.($bCountItems ? ' GROUP BY C.id ':'').'
                        ORDER BY num');
            } break;
            case 2: //свои + общие (выборочное наследование)
            {
                if ($bGetOptions) $sqlSelect = array('C.id','CL.title');
                else {
                    $sqlSelect = array('C.*','CL.*','N.*');
                    if ($bCountItems) $sqlSelect[] = 'COUNT(I.id) as items';
                }
                
                $sqlSelect = join(',', $sqlSelect); 
                
                $sqlWhere[] = 'C.id = N.cat_id';
                $sqlWhere[] = 'N.type_id = '.$oTypeSettings->id;
                if ($bEnabled) $sqlWhere[] = 'N.enabled = 1';
                $aCategories = $this->db->select('SELECT '.$sqlSelect.', (C.type_id != N.type_id) as inherited
                        FROM '.TABLE_AFISHA_CATEGORIES.' C, '.TABLE_AFISHA_CATEGORIES_LANG.' CL, '.TABLE_AFISHA_CATEGORIES_IN.' N
                            '.($bCountItems ? ' LEFT JOIN '.$oTypeSettings->tableName().' I ON I.category_id = N.cat_id ' : '').'     
                        WHERE '.join(' AND ', $sqlWhere).$this->db->langAnd(true, 'C', 'CL').'
                        '.($bCountItems ? ' GROUP BY C.id ':'').'
                        ORDER BY N.num'); 
            } break;
        }
        if ($bGetOptions)
        {
            $sHTML = '';
            if ( ! $mOptions['sel'] && $mOptions['empty']===false) $mOptions['empty'] = true;
            if ( ! empty($mOptions['empty']))
            {
                $sHTML .= '<option value="0">'.(is_string($mOptions['empty']) ? $mOptions['empty'] : _t('', 'не указан')).'</option>';
            }
            foreach ($aCategories as $v) {
                $sHTML .= '<option value="'.$v['id'].'"'.($mOptions['sel']==$v['id']?' selected="selected"':'').'>'.$v['title'].'</option>';
            }
            return $sHTML;
        } else {
            return ( ! bff::adminPanel() ? func::array_transparent($aCategories, 'id', true) : $aCategories);
        }
    }

    /**
     * Инициализация компонента работы с тегами
     * @param integer $typeID ID типа
     * @return AfishaTags
     */
    public function initTags($typeID)
    {
        static $i;
        if (!isset($i)) {
            $i = new AfishaTags();
        }
        $i->setType($typeID);

        return $i;
    }
    
    # календарь
    /**
     * Фиксируем наличие событий за указанную дату
     * @param integer $nItemID id события
     * @param integer $nTypeID id типа события
     * @param string $sAction действие: enabled, disabled, added, removed
     * @return boolean
     */
    function calendarUpdate($nItemID, $nTypeID, $sAction = '')
    {
        return false;
        
        $nIncrement = 1;
        $mDate = '';
        
        $res = $this->db->exec('INSERT INTO '.TABLE_AFISHA_CALENDAR.' (calendar_date, type_id, items) VALUES ('.$mDate.', '.$nTypeID.', 1)
            ON DUPLICATE KEY UPDATE items = items + '.(int)$nIncrement.'
        ');
        
        return !empty($res);
    }
    
    /**
    * 
    */
    function calendarGetActiveDates($mSelectedDate = false, $bJSON = true)
    {
        $n31Days = 2678400; //60*60*24*31
        $nDate = ($mSelectedDate !== false ? (is_string($mSelectedDate) ? strtotime($mSelectedDate) : $mSelectedDate) : time());
        $nDate = mktime(0,0,0,date('m',$nDate),1,date('y',$nDate));
        if ($mSelectedDate!==false) {
            //возвращаем данные за: требуемый месяц
            $nFrom = (int)date('Ymd', $nDate);
            $nTo   = (int)date('Ymd', $nDate + $n31Days);
        } else {
            //возвращаем данные за: предыдущий / текущий / следующий месяца
            $nFrom = (int)date('Ymd', $nDate - $n31Days);
            $nTo   = (int)date('Ymd', $nDate + ($n31Days * 2));
        }

        $res = $this->db->select_one_column('SELECT calendar_date FROM '.TABLE_AFISHA_CALENDAR.' 
            WHERE calendar_date>= '.$nFrom.' AND calendar_date<='.$nTo.' AND items>0');
        if (empty($res)) $res = array();

        return ($bJSON ? '['.join(',',$res).']' : $res);
    }
    
    # RSS
    protected function rssBuild($nLimit = 10)
    {
        // Общее описание
        $sDescription = '
            <title>'._t('afisha','Афиша - городской портал').' '.SITEHOST.'</title>
            <link>'.static::url('index').'</link>
            <language>'.$this->locale->getLanguageSettings(LNG, 'locale').'</language>
            <description>'._t('','Информационный портал области').'</description>
            <image> 
                <url>'.SITEURL_STATIC.'/img/logo.gif</url>
                <title>'._t('','Городской портал').' '.SITEHOST.'</title>
                <link>'.bff::urlBase().'</link>
            </image>
        ';

        $sql = array();
        $sqlLimit = $this->db->prepareLimit(0, $nLimit);
        $aTypes = $this->getTypes();
        foreach ($aTypes as $key=>$v) {
            $sql[] = '(SELECT I.id, IL.title, I.link, IL.content, IL.content_short as description, I.img, I.created, '.$this->db->str2sql($key).' as type
                        FROM '.TABLE_AFISHA_ITEMS_.$key.' I,
                             '.TABLE_AFISHA_ITEMS_LANG.' IL
                        WHERE I.enabled = 1 AND I.moderated = 1 '.$this->model->langAnd($v->id, true, 'I', 'IL').'
                        ORDER BY I.created DESC '.$sqlLimit.')';
        }
        // События
        $aItems = $this->db->select( join(' UNION ', $sql) . ' ORDER BY created DESC ' . $sqlLimit);

        if ( ! empty($aItems))
        {
            foreach ($aItems as $k=>$v)
            {
                $type = $aTypes[$v['type']];
                if ( ! empty($v['img'])) {
                    $imgType = Files::getExtension($v['img']);
                    $imgType = 'image/' . ( $imgType == 'jpg' ? 'jpeg' : $imgType );
                    $v['img'] = $this->getImagesPath(true, array('id'=>$v['id'], 'created'=>$v['created'], 'filename'=>$v['img'], 'size'=>'n'), $type);
                }
                
                //pubDate = Fri, 18 Nov 2011 17:37:20 +0300
                $aItems[$k] = '<item>
                               <title>'.htmlspecialchars(strip_tags($v['title'])).'</title>
                               <link>'.htmlspecialchars(static::url('view', $v['link'])).'</link>
                               <description>'.htmlspecialchars($v['description']).'</description>
                               <yandex:full-text>'.htmlspecialchars($v['content']).'</yandex:full-text>
                               <category>'.htmlspecialchars($type->title).'</category>
                               '.( ! empty($v['img']) ? '<enclosure url="'.$v['img'].'" type="'.$imgType.'"/>' : '').'
                               <pubDate>'.date('D, j M Y G:i:s O', strtotime($v['created'])).'</pubDate>
                               </item>';
            }
        } else {
            $aItems = array();
        }

        return strtr('<?xml version="1.0" encoding="UTF-8"?><rss xmlns:yandex="http://news.yandex.ru" xmlns:media="http://search.yahoo.com/mrss/" version="2.0"><channel>{description}{items}</channel></rss>',
            array(
                '{description}'=>$sDescription, 
                '{items}'=>join('', $aItems)
                )
            );
    }
    
    function rssUpdate($nTypeID)
    {
        // Строим общий файл афиши (для яндекса)
        $sXML = $this->rssBuild(20);
        Files::putFileContent(PATH_PUBLIC.'rss/yandex-afisha.xml', $sXML);
        
        return true;
    }

    /**
     * Обработка ситуации c необходимостью ре-формирования URL
     */
    public function onLinksRebuild()
    {
        $this->model->itemsLinksRebuild();
    }

    /**
     * Формирование списка директорий/файлов требующих проверки на наличие прав записи
     * @return array
     */
    public function writableCheck()
    {
        return array_merge(parent::writableCheck(), array(
            bff::path('afisha', 'images')  => 'dir', # изображения
            bff::path('video'.DS.'afisha') => 'dir', # видео файлы
            PATH_PUBLIC.'rss'              => 'dir', # rss файлы
            PATH_PUBLIC.'rss'.DS.'yandex-afisha.xml' => 'file-e', # yandex-afisha.xml
        ));
    }
}