<?php

use bff\db\Dynprops;

define('TABLE_BBS_ITEMS',                     DB_PREFIX.'bbs_items'); # ОБ
define('TABLE_BBS_ITEMS_FAV',                 DB_PREFIX.'bbs_items_fav'); # избранные ОБ
define('TABLE_BBS_ITEMS_IMAGES',              DB_PREFIX.'bbs_items_images'); # изображения к ОБ
define('TABLE_BBS_ITEMS_COMMENTS',            DB_PREFIX.'bbs_items_comments'); # комментарии к ОБ
define('TABLE_BBS_ITEMS_CLAIMS',              DB_PREFIX.'bbs_items_claims'); # жалобы на ОБ
define('TABLE_BBS_CATEGORIES',                DB_PREFIX.'bbs_categories'); # категории ОБ
define('TABLE_BBS_CATEGORIES_LANG',           DB_PREFIX.'bbs_categories_lang'); # категории ОБ мультиязычные данные
define('TABLE_BBS_CATEGORIES_DYNPROPS',       DB_PREFIX.'bbs_categories_dynprops'); # dp
define('TABLE_BBS_CATEGORIES_DYNPROPS_MULTI', DB_PREFIX.'bbs_categories_dynprops_multi'); # dp.multi
define('TABLE_BBS_CATEGORIES_TYPES',          DB_PREFIX.'bbs_categories_types'); # типы категорий

abstract class BBSBase extends Module implements IModuleWithSvc
{
    /** @var BBSModel */
    public $model = null;
    protected $securityKey = '000000fd4a53a8d1cabce06fe9fe2a70';

    /** @var int Доступность удаление собственных ОБ пользователем:
     * 0|false - удаление пользователю недоступно,
     * 1 - только помечаем как "удалено",
     * 2 - доступно полное удаление
     */
    protected $deleteByUser = 2;

    # Типы дополнительных настроек цены
    const TYPE_PRICE_BART = 2; // бартер
    const TYPE_PRICE_TORG = 4; // торг

    # Статус объявления
    const STATUS_NOTACTIVATED   = 1; // не активировано
    const STATUS_PUBLICATED     = 3; // опубликованное
    const STATUS_PUBLICATED_OUT = 4; // истекший срок публикации
    const STATUS_BLOCKED        = 5; // заблокированное

    # Статус публикации объявления в прессе
    const PRESS_PAYED      = 1; // публикация в прессе оплачена
    const PRESS_PUBLICATED = 2; // опубликовано в прессе

    # ID Услуг
    const SERVICE_UP      = 1; // поднятие
    const SERVICE_MARK    = 2; // выделенние
    const SERVICE_FIX     = 4; // закрепление
    const SERVICE_PREMIUM = 8; // премиум
    const SERVICE_PRESS   = 16; // в прессу
    const SERVICE_VIP     = 524288; // VIP

    public function init()
    {
        parent::init();

        $this->module_title = 'Доска объявлений';

        # инициализируем модуль дин. свойств
        if ( bff::adminPanel() ) {
            if (strpos(bff::$event, 'dynprops')===0)
                $this->dp();
        }
    }

    /**
     * @return BBS
     */
    public static function i()
    {
        return bff::module('bbs');
    }

    /**
     * @return BBSModel
     */
    public static function model()
    {
        return bff::model('bbs');
    }

    /**
     * Формирование URL
     * @param string $key ключ
     * @param mixed $opts параметры
     * @param boolean $dynamic динамическая ссылка
     * @return string
     */
    public static function url($key = '', $opts = array(), $dynamic = false)
    {
        $base = static::urlBase(LNG, $dynamic);
        switch ($key)
        {
            # Просмотр
            case 'view':
                return strtr($opts, array(
                    '{sitehost}' => SITEHOST . bff::locale()->getLanguageUrlPrefix(),
                ));
                break;
            # Добавление
            case 'add':
                return $base.'/add'.static::urlQuery($opts);
                break;
            # Редактирование
            case 'edit':
                return $base.'/edit'.static::urlQuery($opts);
                break;
            # Продвижение
            case 'promote':
                return $base.'/promote'.static::urlQuery($opts);
                break;
            # Поиск
            case 'search':
                $base .= '/search';
                if (empty($opts)) return $base;
                if ( ! is_array($opts)) { $opts = array('cat'=>$opts); }
                return $base.
                    ( ! empty($opts['cat']) ? '/'.$opts['cat'] : '' ).
                    ( ! empty($opts['cat2']) ? '/'.$opts['cat2'] : '' ).
                    static::urlQuery($opts, array('cat', 'cat2'));
                break;
            # Главная
            case 'index':
                return $base.'/'.static::urlQuery($opts);
                break;
        }
        return $base;
    }

    public static function urlView($id, $keyword = '', $cityID = 0)
    {
        if (empty($keyword)) $keyword = 'item';
        return static::urlBase(LNG, true, array('city'=>$cityID)).'/'.$keyword.'-'.$id.'.html';
    }

    /**
     * Описание seo шаблонов страниц
     * @return array
     */
    public function seoTemplates()
    {
        $aTemplates = array(
            'pages' => array(
                'index' => array( // index
                    't'      => 'Главная страница',
                    'macros' => array(
                        'region' => array('t' => 'Регион'),
                    ),
                    'fields' => array(
                        'titleh1' => array(
                            't'      => 'Заголовок H1',
                            'type'   => 'text',
                        ),
                    ),
                ),
                'search' => array( // search
                    't'      => 'Поиск объявлений',
                    'list'   => true,
                    'inherit'=> true,
                    'macros' => array(
                        'category' => array('t' => 'Название категории'),
                        'region' => array('t' => 'Регион'),
                    ),
                    'fields' => array(
                        'titleh1' => array(
                            't'      => 'Заголовок H1',
                            'type'   => 'text',
                        ),
                    ),
                ),
                'add' => array( // add
                    't'      => 'Добавление объявления',
                    'macros' => array(
                        'region' => array('t' => 'Регион'),
                    ),
                    'fields' => array(
                        'titleh1' => array(
                            't'      => 'Заголовок H1',
                            'type'   => 'text',
                        ),
                    ),
                ),
                'view' => array( // view
                    't'      => 'Просмотр объявления',
                    'macros' => array(
                        'title'  => array('t' => 'Заголовок объявления'),
                        'description'  => array('t' => 'Описание (до 150 символов)'),
                        'region' => array('t' => 'Регион'),
                    ),
                    'fields' => array(
                        'share_title'       => array(
                            't'    => 'Заголовок (поделиться в соц. сетях)',
                            'type' => 'text',
                        ),
                        'share_description' => array(
                            't'    => 'Описание (поделиться в соц. сетях)',
                            'type' => 'textarea',
                        ),
                        'share_sitename'    => array(
                            't'    => 'Название сайта (поделиться в соц. сетях)',
                            'type' => 'text',
                        ),
                    ),
                ),
            ),
        );

        return $aTemplates;
    }

    public static function commentsEnabled()
    {
        return (bool)config::sys('bbs.comments.enabled', false);
    }

    /**
     * Инициализация компонента работы с дин. свойствами
     * @return \bff\db\Dynprops объект
     */
    function dp()
    {
        static $oDp = null;
        if (isset($oDp)) return $oDp;

        # подключаем "Динамические свойства"
        $oDp = $this->attachComponent('dynprops',
            new Dynprops('owner_id', TABLE_BBS_CATEGORIES,
                           TABLE_BBS_CATEGORIES_DYNPROPS, TABLE_BBS_CATEGORIES_DYNPROPS_MULTI,
                           1 // полное наследование
                           ) );

        $oDp->setSettings(array(
            'module_name'=>$this->module_name,
            'typesAllowed'=>array(
                Dynprops::typeCheckboxGroup,
                Dynprops::typeRadioGroup,
                Dynprops::typeRadioYesNo,
                Dynprops::typeCheckbox,
                Dynprops::typeSelect,
                Dynprops::typeInputText,
                Dynprops::typeTextarea,
                Dynprops::typeNumber,
                Dynprops::typeRange,
            ),
            'ownerTable_Title' => 'title',
            'cache_method'=>'Bbs_dpSettingsChanged',
            'typesAllowedParent'=>array(Dynprops::typeSelect),
            'datafield_int_last'   => 20,
            'datafield_text_first' => 21,
            'datafield_text_last'  => 30,
            'langs' => $this->locale->getLanguages(false)
        ));

        return $oDp;
    }

    /**
     * Получаем дин. свойства категории
     * @param integer $nCategoryID ID категории
     * @param boolean $bResetCache обнулить кеш
     * @return mixed
     */
    function dpSettings($nCategoryID, $bResetCache = false)
    {
        if ($nCategoryID <= 0) return array();

        $cache = Cache::singleton($this->module_name, 'file');
        $cacheKey = 'cats-dynprops-'.LNG.'-'.$nCategoryID;
        if ($bResetCache) {
            # сбрасываем кеш настроек дин. свойств категории
            return $cache->delete($cacheKey);
        } else {
            if ( ($aSettings = $cache->get($cacheKey)) === false ) { # ищем в кеше
                $aSettings = $this->dp()->getByOwner($nCategoryID, true, true, false);
                $cache->set($cacheKey, $aSettings); # сохраняем в кеш
            }
            return $aSettings;
        }
    }

    /**
     * Метод вызываемый модулем bff\db\Dynprops, в момент изменения настроек дин. свойств категории
     * @param integer $nCategoryID ID категории
     * @param integer $nDynpropID ID дин.свойства
     * @param string $sEvent событие, генерирующее вызов метода
     * @return mixed
     */
    function dpSettingsChanged($nCategoryID, $nDynpropID, $sEvent)
    {
        if (empty($nCategoryID)) return false;
        $this->dpSettings($nCategoryID, true);
    }

    /**
     * Формирование SQL запроса для сохранения дин.свойств
     * @param integer $nCategoryID ID категории
     * @param string $sFieldname ключ в $_POST массиве
     * @return array
     */
    function dpSave($nCategoryID, $sFieldname = 'd')
    {
        $aData = $this->input->post($sFieldname, TYPE_ARRAY);

        $aDynpropsData = array();
        foreach ($aData as $props) {
            foreach ($props as $id=>$v) {
                $aDynpropsData[$id] = $v;
            }
        }

        $aDynprops = $this->dp()->getByID(array_keys($aDynpropsData), true);

        return $this->dp()->prepareSaveDataByID($aDynpropsData, $aDynprops, 'update', true);
    }

    /**
     * Формирование формы редактирования / фильтра дин.свойств
     * @param integer $nCategoryID ID категории
     * @param boolean $bSearch формирование формы поиска
     * @param array|boolean $aData данные или FALSE
     * @param array $aExtra доп.данные
     * @return string HTML template
     */
    function dpForm($nCategoryID, $bSearch = true, $aData = false, $aExtra = array())
    {
         if (empty($nCategoryID)) return '';

         if ($bSearch) {
            if ( ! bff::adminPanel()) {
                // формируем форму дин. свойств
                $aDynpropsData = array();
                if ( ! empty($aData)) {
                    // для поиска - раскладываем с ключем 'f' для формирования child-свойств
                    foreach ($aData['f'] as $k=>$v) { $aDynpropsData['f'.$k] = $v; }
                    foreach ($aData['fc'] as $k=>$v) { $aDynpropsData['f'.$k] = $v; }
                }
                $aForm = $this->dp()->form($nCategoryID, $aDynpropsData, true, true, 'f', 'dynprops.search', $this->module_dir_tpl, $aExtra);
            } else {
                $aForm = $this->dp()->form($nCategoryID, $aData, true, true, 'd', 'search.inline', false, $aExtra);
            }
         } else {
            if ( ! bff::adminPanel()) {
                $aForm = $this->dp()->form($nCategoryID, $aData, true, false, 'd', 'dynprops.form', $this->module_dir_tpl, $aExtra);
            } else {
                $aForm = $this->dp()->form($nCategoryID, $aData, true, false, 'd', 'form.table', false, $aExtra);
            }
         }
         return ( ! empty($aForm['form']) ? $aForm['form'] : '');
    }

    /**
     * Отображение дин. свойств
     * @param integer $nCategoryID ID категории
     * @param array $aData данные
     * @param string $sKey ключ
     */
    function dpView($nCategoryID, $aData, $sKey = 'd')
    {
        if ( ! bff::adminPanel()) {
            $aForm = $this->dp()->form($nCategoryID, $aData, true, false, $sKey, 'dynprops.view', $this->module_dir_tpl);
        } else {
            $aForm = $this->dp()->form($nCategoryID, $aData, true, false, $sKey, 'view.table');
        }

        return ( ! empty($aForm['form']) ? $aForm['form'] : '');
    }

    /**
     * Инициализация компонента BBSItemImages
     * @param mixed $nItemID ID объявления
     * @return BBSItemImages component
     */
    public function itemImages($nItemID = false)
    {
        static $i;
        if (!isset($i)) {
            require_once $this->module_dir.'bbs.item.images.php';
            $i = new BBSItemImages();
        }
        $i->setRecordID($nItemID);
        return $i;
    }

    /**
     * Инициализация компонента BBSItemComments
     * @return BBSItemComments component
     */
    public function itemComments()
    {
        static $c;
        if (!isset($c)) {
            include_once $this->module_dir.'bbs.item.comments.php';
            $c = new BBSItemComments();
        }
        return $c;
    }

    /**
     * Включена ли премодерация объявлений
     * @return boolean
     */
    public static function premoderation()
    {
        return (bool)config::sys('bbs.premoderation', true);
    }

    /**
     * Удаление ОБ
     */
    function itemDelete($nItemID)
    {
        $this->itemImages($nItemID)->deleteAllImages();
        $res = $this->model->itemDelete($nItemID);
        if ( ! empty($res) ) {
            return true;
        } else {
            return false;
        }
    }

    /**
     * Обрабатываем параметры запроса на добавление/редактирование ОБ
     * @param integer $nItemID ID объявления или 0
     * @param boolean $bSubmit выполняем сохранение/редактирование
     * @return array параметры
     */
    function validateItemData($nItemID, $bSubmit)
    {
        $aData = $this->input->postm(array(
            'cat_id'   => TYPE_UINT, // Категория
            'cat_type' => TYPE_UINT, // Тип
            'title'    => array(TYPE_NOTAGS, 'len'=>150), // Название
            'descr'    => array(TYPE_NOTAGS, 'len'=>10000), // Описание
            'price'        => TYPE_PRICE, // Цена
            'price_params' => TYPE_ARRAY_UINT, // Цена: параметры; торг...
            'price_curr'   => TYPE_UINT, // Цена: валюта
            'city_id'      => TYPE_UINT, // Город
            'addr_addr'    => array(TYPE_NOTAGS, 'len'=>200), // Адрес
            'user_name'    => array(TYPE_NOTAGS, 'len'=>100), // имя
            'user_phone'   => array(TYPE_NOTAGS, 'len'=>50), // телефон
            'addr_lat' => TYPE_NUM, // адрес, координата LAT
            'addr_lng' => TYPE_NUM, // адрес, координата LNG
            'company_id' => TYPE_UINT, //Объявление кампании
            'user_type' => TYPE_UINT,   //частное лицо/организация
        ));

        if ($bSubmit)
        {
            # Категория
            $nCategoryID = $aData['cat_id'];
            if ( ! $nCategoryID ) {
                $this->errors->set( _t('bbs','Выберите категорию') );
            } else {
                # проверяем наличие подкатегорий
                $aCat = $this->model->categoryData($nCategoryID);
                if ( $aCat['subs'] > 0 ) {
                    $this->errors->set( _t('bbs','Выбранная категория не должна содержать подкатегории') );
                } else {
                    # сохраняем ID категорий(parent и текущей), для возможности дальнейшего поиска по ним
                    $nParentsID = $this->model->categoryParentsID($nCategoryID, true);
                    foreach ($nParentsID as $lvl=>$id) {
                        $aData['cat_id'.$lvl] = $id;
                    }
                }
            }

            if (empty($aData['title'])) {
                $this->errors->set( _t('bbs', 'Укажите заголовок объявления'), 'title' );
            } elseif (mb_strlen($aData['title'])<5) {
                $this->errors->set( _t('bbs', 'Заголовок слишком короткий'), 'title' );
            } else {
                # формируем URL-keyword на основе title
                $aData['keyword'] = mb_strtolower( func::translit( $aData['title'] ) );
                $aData['keyword'] = preg_replace('/[^a-z0-9_\-]/', '', $aData['keyword'] );
            }

            # чистим описание, дополнительно
            $aData['descr'] = $this->input->cleanTextPlain($aData['descr'], false, false);
            if (mb_strlen($aData['descr'])<12) {
                $this->errors->set( _t('bbs', 'Описание слишком короткое'), 'descr' );
            }

            # город
            if ( ! Geo::cityIsValid($aData['city_id']) ) {
                $this->errors->set( _t('bbs', 'Город указан некорректно') );
            }

            # проквочиваем заголовок
            $aData['title_edit'] = $aData['title'];
            $aData['title'] = HTML::escape($aData['title_edit']);

            // конвертируем цену в цену для поиска
            $aData['price_search'] = bff::convertPriceForSearch($aData['price'], $aData['price_curr']);

            if ( $this->errors->no() ) {
                # формируем полный адрес
                $aData['addr_full'] = 'г.'.Geo::regionTitle($aData['city_id']) .
                                        ( ! empty($aData['addr_addr']) && $aCat['addr']
                                           ? ', '.$aData['addr_addr'] : '' );
            }
            # проверим company_id
            if (bff::adminPanel()) {
                unset($aData['company_id']);
            } else {
                if (User::id() && $aData['user_type'] == 2 && $aData['company_id'] > 0) {
                    $aComp = Users::model()->companyInfo(User::id(), $aData['company_id']);
                    if (empty($aComp)) {
                        $aData['company_id'] = 0;
                    }
                } else {
                    $aData['company_id'] = 0;
                }
            }
            unset($aData['user_type']);

        }
        return $aData;
    }

    function getPublicatePeriods($nFrom = false, $mDatesFormat = 'd.m.Y')
    {
        if ($nFrom === false) $nFrom = BFF_NOW;

        $nDay  = 86400;
        $nWeek = 604800; // $nDay * 7

        $aVariants = array(
            //1 => array('id'=>1, 't'=>'3 дня',    'p'=>$nFrom+259200),
            //2 => array('id'=>2, 't'=>'1 неделю', 'p'=>$nFrom+$nWeek),
            //3 => array('id'=>3, 't'=>'2 недели', 'p'=>$nFrom+$nWeek*2),
            4 => array('id'=>4, 't'=>'3 недели', 'p'=>$nFrom+$nWeek*3),
            5 => array('id'=>5, 't'=>'1 месяц',  'p'=>strtotime('+1 month', $nFrom)),
            //6 => array('id'=>6, 't'=>'2 месяца', 'p'=>strtotime('+2 months', $nFrom)),
        );

        if ($mDatesFormat!==false) {
            foreach ($aVariants as $k=>$v) {
                $aVariants[$k]['date'] = date($mDatesFormat, $v['p']);
            }
        }

        reset($aVariants);

        return array(
            'options' => HTML::selectOptions($aVariants, 0, false, 'id', 't'),
            'variants'=> $aVariants);
    }

    /**
     * Получаем дату окончания срока публикации ОБ
     * @param mixed $nPeriod id периода или FALSE - первый их доступных
     * @param mixed $nFrom время отсчета срока публикации, false - NOW
     * @return string дата окончания срока публикации ОБ
     */
    function getPublicatePeriodDate($nPeriod = false, $nFrom = false)
    {
        $aPeriods = $this->getPublicatePeriods($nFrom, 'Y-m-d H:i:s');
        $aPeriods = $aPeriods['variants'];
        if ($nPeriod === false || !isset($aPeriods[$nPeriod])) {
            reset($aPeriods);
            $nPeriod = key($aPeriods);
        }
        return $aPeriods[$nPeriod]['date'];
    }

    /**
     * Получаем срок публикации объявления в днях
     * @param mixed $mFrom дата, от которой выполняется подсчет срока публикации
     * @param string $mFormat тип требуемого результата, строка = формат даты, false - unixtime
     * @return int
     */
    public function getItemPublicationPeriod($mFrom = false, $mFormat = 'Y-m-d H:i:s')
    {
        $nDays = intval(config::get('bbs_item_publication_period'));
        if ($nDays <= 0) {
            $nDays = 7;
        }

        if (empty($mFrom) || is_bool($mFrom)) {
            $mFrom = strtotime($this->db->now());
        } else if (is_string($mFrom)) {
            $mFrom = strtotime($mFrom);
            if ($mFrom === false) {
                $mFrom = strtotime($this->db->now());
            }
        }

        $nPeriod = strtotime('+' . $nDays . ' days', $mFrom);
        if (!empty($mFormat)) {
            return date($mFormat, $nPeriod);
        } else {
            return $nPeriod;
        }
    }

    /**
     * Получаем срок продления объявления в днях
     * @param mixed $mFrom дата, от которой выполняется подсчет срока публикации
     * @param string $mFormat тип требуемого результата, строка = формат даты, false - unixtime
     * @return int
     */
    public function getItemRefreshPeriod($mFrom = false, $mFormat = 'Y-m-d H:i:s')
    {
        $nDays = intval(config::get('bbs_item_refresh_period'));
        if ($nDays <= 0) {
            $nDays = 7;
        }

        if (empty($mFrom) || is_bool($mFrom)) {
            $mFrom = $this->db->now();
        }
        if (is_string($mFrom)) {
            $mFrom = strtotime($mFrom);
            if ($mFrom === false) {
                $mFrom = strtotime($this->db->now());
            }
        }
        $nPeriod = strtotime('+' . $nDays . ' days', $mFrom);
        if (!empty($mFormat)) {
            return date($mFormat, $nPeriod);
        } else {
            return $nPeriod;
        }
    }



    /**
     * Получаем дату окончания срока активации ОБ
     * @return string дата окончания срока публикации ОБ
     */
    function getActivateExpireDate()
    {
        $aData = $this->configLoad();
        $nHours = 24;
        if( ! empty($aData['item_delete_unactivate_period'])){
            $nHours = $aData['item_delete_unactivate_period'];
        }
        return date('Y-m-d H:i:s', strtotime('+'.$nHours.' hours'));
    }

    /**
     * Причины жалобы на ОБ
     * @return array
     */
    function getItemClaimReasons()
    {
        return array(
            1  => _t('','Неверная контактная информация'),
            2  => _t('','Объявление не соответствует рубрике'),
            4  => _t('','Объявление не отвечает правилам портала'),
            8  => _t('','Некорректная фотография'),
            16 => _t('','Антиреклама/дискредитация'),
            32 => _t('','Другое'),
        );
    }

    /**
     * Формирование текста причины жалобы на ОБ
     * @param int $nReasons причины
     * @param string $sComment дополнительный комментарий пользователя
     * @return string
     */
    function getItemClaimText($nReasons, $sComment)
    {
        $reasons = $this->getItemClaimReasons();
        if ( ! empty($nReasons) && !empty($reasons))
        {
            $r_text = array();
            foreach ($reasons as $rk=>$rv) {
                if ($rk!=32 && $rk & $nReasons) {
                    $r_text[] = $rv;
                }
            }
            $r_text = join(', ', $r_text);
            if ($nReasons & 32 && !empty($sComment)) {
                $r_text .= ', '.$sComment;
            }
            return $r_text;
        }
        return '';
    }

    function getPriceParams()
    {
        return array(
            //self::TYPE_PRICE_BART => _t('','бартер'),
            self::TYPE_PRICE_TORG => _t('','возможен торг'),
        );
    }

    /**
     * Метод обрабатывающий событие "блокировки/разблокировки пользователя"
     * @param integer $nUserID ID пользователя
     * @param boolean $bBlocked true - заблокирован, false - разблокирован
     */
    function onUserBlocked($nUserID, $bBlocked)
    {
        if ($bBlocked)
        {
            // при блокировке пользователя -> блокируем все его объявления
            $aItems = $this->model->itemsDataByFilter(
                                            array('user_id'=>$nUserID,
                                                  'status IN('.self::STATUS_PUBLICATED.', '.self::STATUS_PUBLICATED_OUT.')',
                                                  'deleted'=>0),
                                            array('id'));
            if ( ! empty($aItems)) {
                $this->model->itemsSave( array_keys( $aItems ), array(
                    'blocked_num = blocked_num + 1',
                    'status_prev = status',
                    'status' => self::STATUS_BLOCKED,
                    'moderated' => 1,
                    'blocked_reason' => _t('bbs', 'Аккаунт пользователя заблокирован'),
                ));
            }
        } else {
            // при разблокировке -> разблокируем
            $aItems = $this->model->itemsDataByFilter(
                                            array('user_id'=>$nUserID,
                                                  'status'=>self::STATUS_BLOCKED,
                                                  'moderated'=>1,
                                                  'deleted'=>0),
                                            array('id'));
            if ( ! empty($aItems)) {
                $this->model->itemsSave( array_keys( $aItems ), array(
                    'status = status_prev', // возвращаем предыдущий статус
                    'status_prev' => self::STATUS_BLOCKED,
                    //'blocked_reason' => '', // оставляем последнюю причину блокировки
                ));
            }
        }
    }

    /**
     * Метод обрабатывающий событие "активации пользователя"
     * @param integer $nUserID ID активированного пользователя
     */
    function onUserActivated($nUserID)
    {
        /**
         * Активируем объявления, добавленные пользователем и еще неактивированные
         */
        $aItems = $this->model->itemsDataByFilter(
            array('user_id'=>$nUserID, 'status'=>self::STATUS_NOTACTIVATED),
            array('id')
        );
        if ( ! empty($aItems) ) {
            $this->model->itemsSave(array_keys($aItems), array('status'=>self::STATUS_PUBLICATED, 'activate_expire' => ''));
        }
    }

    # --------------------------------------------------------
    # Активация услуг

    function svcActivate($nItemID, $nSvcID, $aSvcData = false, array &$aSvcSettings = array())
    {
        $svc = $this->svc();
        if ( ! $nSvcID ) {
            $this->errors->set(_t('svc', 'Неудалось активировать услугу'));
            return false;
        }
        if ( empty($aSvcData) ) {
            $aSvcData = $svc->model->svcData($nSvcID);
            if ( empty($aSvcData) ) {
                $this->errors->set(_t('svc', 'Неудалось активировать услугу'));
                return false;
            }
        }

        // получаем данные об объявлении
        if ( empty($aItemData) ) {
            $aItemData = $this->model->itemData($nItemID, array(
                'id','status','deleted', // ID, статус, флаг "ОБ удалено"
                'publicated_to',# дата окончания публикации
                'svc',          # битовое поле активированных услуг
                'up_activate',  # кол-во оставшихся оплаченных поднятий (оплаченных пакетно)
                'cat_id1',      # основная категория (для подсчета позиции ОБ в данной категории)
                'fixed_to',     # дата окончания "Закрепления"
                'vip_to',       # дата окончания "VIP"
                'premium_to',   # дата окончания "Премиум"
                'marked_to',    # дата окончания "Выделение"
                'press_status', # статус "Печать в прессе"
            ));
        }

        // проверяем статус объявления
        if ( empty($aItemData) || $aItemData['status'] != self::STATUS_PUBLICATED || $aItemData['deleted'] ) {
            $this->errors->set( _t('bbs', 'Для указанного объявления невозможно активировать данную услугу') );
            return false;
        }

        // активируем пакет услуг
        if ($aSvcData['type'] == Svc::TYPE_SERVICEPACK)
        {
            $aServices = ( isset($aSvcData['svc']) ? $aSvcData['svc'] : array() );
            if ( empty($aServices) ) {
                $this->errors->set(_t('bbs', 'Неудалось активировать пакет услуг'));
                return false;
            }
            $aServicesID = array(); foreach ($aServices as $k=>$v) $aServicesID[] = $v['id'];
            $aServices = $svc->model->svcData($aServicesID, array('*'));
            if ( empty($aServices) ) {
                $this->errors->set(_t('bbs', 'Неудалось активировать пакет услуг'));
                return false;
            }

            // проходимся по услугам, входящим в пакет
            // активируем каждую из них
            $nSuccess = 0;
            foreach ($aServices as $k=>$v)
            {
                $v['cnt'] = $aSvcData['svc'][$k]['cnt'];
                $res = $this->svcActivateService($nItemID, $v['id'], $v, $aItemData, true, $aSvcSettings);
                if ( $res ) {
                    switch ($v['id'])
                    {
                        case self::SERVICE_PRESS: {
                            // отправляем email-уведомление о скорой публикации в прессе
                            // --
                        } break;
                    }
                    $nSuccess++;
                }
            }
            return true;
        } else {
            // активируем услугу
            return $this->svcActivateService($nItemID, $nSvcID, $aSvcData, $aItemData, false, $aSvcSettings);
        }
    }

    /**
     * Активация услуги для Объявления
     * @param integer $nItemID ID объявления
     * @param integer $nSvcID ID услуги
     * @param mixed $aSvcData данные об услуге(*) или FALSE
     * @param mixed $aItemData @ref данные об объявлении или FALSE
     * @param boolean $bFromPack услуга активируется из пакета услуг
     * @param array $aSvcSettings @ref дополнительные параметры услуги/нескольких услуг
     * @return boolean true - услуга успешно активирована, false - ошибка активации услуги
     */
    protected function svcActivateService($nItemID, $nSvcID, $aSvcData = false, &$aItemData = false, $bFromPack = false, array &$aSvcSettings = array())
    {
        if ( empty($nItemID) || empty($aItemData) || empty($nSvcID) ) {
            $this->errors->set(_t('svc', 'Неудалось активировать услугу'));
            return false;
        }
        $svc = $this->svc();
        if ( empty($aSvcData) ) {
            $aSvcData = $svc->model->svcData($nSvcID);
            if ( empty($aSvcData) ) {
                $this->errors->set(_t('svc', 'Неудалось активировать услугу'));
                return false;
            }
        }

        $sNow = $this->db->now();
        $nDay = 86400;
        $publicatedTo = strtotime($aItemData['publicated_to']);
        $aUpdate = array();
        switch ($nSvcID)
        {
            case self::SERVICE_UP: // "поднятие"
            {
                if ( $bFromPack ) {
                    $nPosition = $this->model->itemPositionInCategory($nItemID, $aItemData['cat_id1']);
                    if ( $nPosition === 1 ) {
                        // если ОБ находится на первой позиции в основной категории
                        // НЕ выполняем "поднятие", только помечаем доступное для активации кол-во поднятий
                        $aUpdate['up_activate'] = $aSvcData['cnt'];
                        break;
                    }
                    // при "поднятии" пакетно помечаем доступное для активации кол-во "поднятий"
                    // -1 ("поднятие" при активации пакета услуг)
                    $aUpdate['up_activate'] = ($aSvcData['cnt'] - 1);
                } else {
                    // если есть неиспользованные "поднятия", используем их
                    if ( ! empty($aItemData['up_activate']) ) {
                        $aUpdate['up_activate'] = ($aItemData['up_activate'] - 1);
                    }
                }
                $aUpdate['publicated_order'] = $sNow;
            } break;
            case self::SERVICE_MARK: // "выделение"
            {
                $nDays = 7; // период действия услуги (в днях)
                if ( $bFromPack && ! empty($aSvcData['cnt']) ) {
                    // при пакетной активации, период действия берем из настроек пакета услуг
                    $nDays = $aSvcData['cnt'];
                }

                // считаем дату окончания действия услуги
                $to = strtotime('+'.$nDays.' days', (
                                // если услуга уже активна => продлеваем срок действия
                                ($aItemData['svc'] & $nSvcID) ? strtotime($aItemData['marked_to']) :
                                // если неактивна => активируем на требуемый период от текущей даты
                                time()
                            ));
                $toStr = date('Y-m-d H:i:s', $to);
                // в случае если дата публикация объявления завершается раньше окончания услуги:
                if ($publicatedTo < $to) {
                    // продлеваем публикацию
                    $aUpdate['publicated_to'] = $toStr;
                }
                // помечаем срок действия услуги
                $aUpdate['marked_to'] = $toStr;
                // помечаем активацию услуги
                $aUpdate['svc'] = ($aItemData['svc'] | $nSvcID);
            } break;
            case self::SERVICE_FIX: // "закрепление"
            {
                $nDays = 7; // период действия услуги (в днях)

                // считаем дату окончания действия услуги
                $to = strtotime('+'.$nDays.' days', (
                                // если услуга уже активна => продлеваем срок действия
                                ($aItemData['svc'] & $nSvcID) ? strtotime($aItemData['fixed_to']) :
                                // если неактивна => активируем на требуемый период от текущей даты
                                time()
                            ));
                $toStr = date('Y-m-d H:i:s', $to);
                // в случае если дата публикация объявления завершается раньше окончания услуги:
                if ($publicatedTo < $to) {
                    // продлеваем публикацию
                    $aUpdate['publicated_to'] = $toStr;
                }
                // помечаем срок действия услуги
                $aUpdate['fixed_to'] = $toStr;
                // ставим выше среди закрепленных
                $aUpdate['fixed_order'] = $sNow;
                // помечаем активацию услуги
                $aUpdate['svc'] = ($aItemData['svc'] | $nSvcID);
            } break;
            case self::SERVICE_PREMIUM: // премиум
            {
                $nDays = 7; // период действия услуги "Премиум" (в днях)
                if ( $bFromPack && ! empty($aSvcData['cnt']) ) {
                    // при пакетной активации, период действия берем из настроек пакета услуг
                    $nDays = $aSvcData['cnt'];
                }

                // считаем дату окончания действия услуги
                $to = strtotime('+'.$nDays.' days', (
                                // если услуга уже активна => продлеваем срок действия
                                ($aItemData['svc'] & $nSvcID) ? strtotime($aItemData['premium_to']) :
                                // если неактивна => активируем на требуемый период от текущей даты
                                time()
                            ));
                $toStr = date('Y-m-d H:i:s', $to);
                // в случае если дата публикация объявления завершается раньше окончания услуги:
                if ($publicatedTo < $to) {
                    // продлеваем публикацию
                    $aUpdate['publicated_to'] = $toStr;
                }
                // помечаем срок действия услуги
                $aUpdate['premium_to'] = $toStr;
                // ставим выше среди премиум
                $aUpdate['premium_order'] = $sNow;
                // помечаем активацию услуги
                $aUpdate['svc'] = ($aItemData['svc'] | $nSvcID);
            } break;
            case self::SERVICE_PRESS: // в прессу
            {
                switch ($aItemData['press_status'])
                {
                    case self::PRESS_PAYED: {
                        if ( ! $bFromPack ) $this->errors->set( _t('svc', 'Объявление будет опубликовано в ближайшее время') );
                        return false;
                    } break;
                    case self::PRESS_PUBLICATED: {
                        if ( ! $bFromPack ) $this->errors->set( _t('svc', 'Объявление уже опубликовано прессе') );
                        return false;
                    } break;
                    default: {
                        // помечаем на "Публикацию в прессе"
                        $aUpdate['press_status'] = self::PRESS_PAYED;
                    } break;
                }
            } break;
            case self::SERVICE_VIP: // "vip"
            {
                $nDays = 7; // период действия услуги (в днях)

                // считаем дату окончания действия услуги
                $to = strtotime('+'.$nDays.' days', (
                    // если услуга уже активна => продлеваем срок действия
                ($aItemData['svc'] & $nSvcID) ? strtotime($aItemData['vip_to']) :
                    // если неактивна => активируем на требуемый период от текущей даты
                    time()
                ));
                $toStr = date('Y-m-d H:i:s', $to);
                // в случае если дата публикация объявления завершается раньше окончания услуги:
                if($publicatedTo < $to) {
                    // продлеваем публикацию
                    $aUpdate['publicated_to'] = $toStr;
                }
                // помечаем срок действия услуги
                $aUpdate['vip_to'] = $toStr;
                // помечаем активацию услуги
                $aUpdate['svc'] = ($aItemData['svc'] | $nSvcID);
            } break;

        }

        $res = $this->model->itemSave($nItemID, $aUpdate);
        if ( ! empty($res) ) {
            // актуализируем данные об объявлении
            // для корректной пакетной активации услуг
            if ( ! empty($aUpdate) ) {
                foreach ($aUpdate as $k=>$v) {
                    $aItemData[$k] = $v;
                }
            }
            return true;
        }
        return false;
    }

    function svcBillDescription($nItemID, $nSvcID, $aData = false, array &$aSvcSettings = array())
    {
        $aSvc = ( ! empty($aData['svc']) ? $aData['svc'] :
                    $this->svc()->model->svcData($nSvcID) );

        $aItem = ( ! empty($aData['item']) ? $aData['item'] :
                    $this->model->itemData($nItemID, array('id','link','title')) );

        $sItemLink = static::url('view', $aItem['link']);
        list($sLinkOpen, $sLinkClose) = ( ! empty($sItemLink) ? array('<a href="'.$sItemLink.'" class="bill-bbs-item-link" data-item="'.$nItemID.'">', '</a>') : array('','') );

        if ($aSvc['type'] == Svc::TYPE_SERVICE)
        {
            switch ($nSvcID)
            {
                case self::SERVICE_UP:      { return _t('bbs', 'Поднятие [a]объявления[b] в списке', array('a'=>$sLinkOpen,'b'=>$sLinkClose)); } break;
                case self::SERVICE_MARK:    { return _t('bbs', 'Выделение [a]объявления[b] цветом', array('a'=>$sLinkOpen,'b'=>$sLinkClose)); } break;
                case self::SERVICE_FIX:     { return _t('bbs', 'Закрепление [a]объявления[b]', array('a'=>$sLinkOpen,'b'=>$sLinkClose)); } break;
                case self::SERVICE_PREMIUM: { return _t('bbs', 'Премиум размещение [a]объявления[b]', array('a'=>$sLinkOpen,'b'=>$sLinkClose)); } break;
                case self::SERVICE_PRESS:   { return _t('bbs', 'Размещение [a]объявления[b] в прессе', array('a'=>$sLinkOpen,'b'=>$sLinkClose)); } break;
            }
        } else if ($aSvc['type'] == Svc::TYPE_SERVICEPACK) {
            return _t('bbs', 'Пакет услуг "[pack]" для [a]объявления[b]',
                array(
                    'pack'=>$aSvc['title'],
                    'a'=>$sLinkOpen,'b'=>$sLinkClose));
        }
    }

    function svcIsVIPEnabled(&$aData = array())
    {
        static $cache;
        if( ! isset($cache) ) {
            $cache = Svc::model()->svcData(self::SERVICE_VIP);
        }
        $aData = $cache;
        return ! empty( $cache['on'] );
    }


    function svcCron()
    {
        if ( ! bff::cron() ) return;

        $sNow = $this->db->now();
        $sEmpty = '0000-00-00 00:00:00';

        # Деактивируем услугу "Выделение"
        $this->db->exec('UPDATE '.TABLE_BBS_ITEMS.'
            SET svc = (svc - '.self::SERVICE_MARK.'), marked_to = :empty
            WHERE (svc & '.self::SERVICE_MARK.') AND marked_to <= :now',
            array(':now'=>$sNow,':empty'=>$sEmpty));

        # Деактивируем услугу "Закрепление"
        $this->db->exec('UPDATE '.TABLE_BBS_ITEMS.'
            SET svc = (svc - '.self::SERVICE_FIX.'), fixed_to = :empty, fixed_order = :empty
            WHERE (svc & '.self::SERVICE_FIX.') AND fixed_to <= :now',
            array(':now'=>$sNow,':empty'=>$sEmpty));

        # Деактивируем услугу "Премиум"
        $this->db->exec('UPDATE '.TABLE_BBS_ITEMS.'
            SET svc = (svc - '.self::SERVICE_PREMIUM.'), premium_to = :empty, premium_order = :empty
            WHERE (svc & '.self::SERVICE_PREMIUM.') AND premium_to <= :now',
            array(':now'=>$sNow,':empty'=>$sEmpty));
    }

    /**
     * Обработка ситуации c необходимостью ре-формирования URL
     */
    public function onLinksRebuild()
    {
        $this->model->itemsLinksRebuild();
    }

    /**
     * Формирование списка директорий/файлов требующих проверки на наличие прав записи
     * @return array
     */
    public function writableCheck()
    {
        return array_merge(parent::writableCheck(), array(
            bff::path('board', 'images') => 'dir', # изображения
            bff::path('tmp', 'images')   => 'dir', # tmp
        ));
    }
}