<?php

use bff\db\NestedSetsTree;

class BBSModel extends Model
{
    /** @var BBS */
    protected $controller;

    /** @var NestedSetsTree для категорий */
    var $treeCategories;
    var $langCategories = array(
        'title'         => TYPE_NOTAGS, # Название
        'titleh1'       => TYPE_NOTAGS, # Заголовок H1
        'mtitle'        => TYPE_NOTAGS, # Meta Title
        'mkeywords'     => TYPE_NOTAGS, # Meta Keywords
        'mdescription'  => TYPE_NOTAGS, # Meta Description
    );

    var $langCategoriesTypes = array(
        'title'        => TYPE_STR, # название
    );

    function init()
    {
        parent::init();

        # подключаем nestedSets категории
        $this->treeCategories = new NestedSetsTree(TABLE_BBS_CATEGORIES);
        $this->treeCategories->init();
    }

    # --------------------------------------------------------------------
    # Объявления

    /**
     * Список объявлений (admin)
     * @param array $aFilter фильтр списка объявлений
     * @param bool $bCount только подсчет кол-ва объявлений
     * @param array $aBind подстановочные данные
     * @param string $sqlLimit
     * @param string $sqlOrder
     * @return mixed
     */
    function itemsListing(array $aFilter, $bCount = false, array $aBind = array(), $sqlLimit = '', $sqlOrder = '') //admin
    {
        $aFilter = $this->prepareFilter($aFilter, 'I', $aBind);
        if ($bCount) {
            return $this->db->one_data('SELECT COUNT(I.id) FROM '.TABLE_BBS_ITEMS.' I '.$aFilter['where'], $aFilter['bind']);
        }

        return $this->db->select('SELECT I.id, I.link, I.created, I.deleted, I.moderated, I.title
               FROM '.TABLE_BBS_ITEMS.' I
               '.$aFilter['where']
               .( ! empty($sqlOrder) ? ' ORDER BY '.$sqlOrder : '')
               .$sqlLimit, $aFilter['bind'] );
    }

    /**
     * Список объявлений (фронтенд)
     * @param array $aFilter фильтр списка объявлений
     * @param bool $bCount только подсчет кол-ва объявлений
     * @param string $sqlLimit
     * @param string $sqlOrder
     * @return mixed
     */
    function itemsList(array $aFilter = array(), $bCount = false, $sqlLimit = '', $sqlOrder = '') # frontend
    {
        $aFilter[':cat'] = 'I.cat_id = C.id';
        $aFilter[':deleted'] = 'I.deleted = 0';
        $aFilter[':catlang'] = $this->db->langAnd(false, 'C', 'CL');
        if (BBS::premoderation() && ! isset($aFilter[':moderated'])) {
            $aFilter[':moderated'] = 'I.moderated > 0';
        }

        $aFilter = $this->prepareFilter($aFilter, 'I');

        if ($bCount) {
            return $this->db->one_data('SELECT COUNT(I.id) FROM '.TABLE_BBS_ITEMS.' I, '.TABLE_BBS_CATEGORIES.' C, '.TABLE_BBS_CATEGORIES_LANG.' CL'.$aFilter['where'], $aFilter['bind'] );
        }

        $bJoinFavorites = (User::id() && $this->security->userCounter('bbs_fav') > 0);

        $aData = $this->db->select('SELECT I.id, I.link, I.title, I.img_m, I.imgcnt, I.addr_full,
                                    I.price, I.price_curr, I.price_params,
                                    I.svc, (I.svc & '.BBS::SERVICE_FIX.') as fixed, (I.svc & '.BBS::SERVICE_MARK.') as marked,
                                    CL.title as cat_title, C.prices as cat_prices,
                                    '.($bJoinFavorites ? ' (F.item_id)' : '0').' as fav
                                  FROM '.TABLE_BBS_ITEMS.' I
                                     '.($bJoinFavorites ? ' LEFT JOIN '.TABLE_BBS_ITEMS_FAV.' F ON I.id = F.item_id AND F.user_id = '.User::id() : '').',
                                     '.TABLE_BBS_CATEGORIES.' C, '.TABLE_BBS_CATEGORIES_LANG.' CL
                                  '.$aFilter['where'].'
                                  '.( ! empty($sqlOrder) ? ' ORDER BY '.$sqlOrder : '').'
                                  '.$sqlLimit, $aFilter['bind']);

        if ( ! empty($aData) )
        {
            //
        }

        return $aData;
    }

    /**
     * Список избранных объявлений пользователя (фронтенд)
     * @param integer $nUserID ID пользователя
     * @param bool $bCount только подсчет кол-ва объявлений
     * @return array
     */
    function itemsListFav($nUserID, $bCount = false) # frontend
    {
        $bCurrentUser = User::isCurrent($nUserID);
        $aBind = array(':user'=>$nUserID);

        if ($bCount) {
            if ($bCurrentUser) {
                return $this->security->userCounter('bbs_fav');
            } else {
                return $this->db->one_data('SELECT COUNT(F.item_id) FROM '.TABLE_BBS_ITEMS_FAV.' F WHERE F.user_id = :user', $aBind );
            }
        }

        if ($bCurrentUser && ! $this->security->userCounter('bbs_fav')) {
            return array();
        }

        $aBind[':status'] = BBS::STATUS_PUBLICATED;
        return $this->db->select('SELECT I.id, I.link, I.title, I.img_m, I.imgcnt, I.addr_full,
                                    I.price, I.price_curr, I.price_params,
                                    CL.title as cat_title, C.prices as cat_prices,
                                    I.svc, (I.svc & '.BBS::SERVICE_FIX.') as fixed, (I.svc & '.BBS::SERVICE_MARK.') as marked
                                  FROM '.TABLE_BBS_ITEMS.' I,
                                       '.TABLE_BBS_CATEGORIES.' C,
                                       '.TABLE_BBS_CATEGORIES_LANG.' CL,
                                       '.TABLE_BBS_ITEMS_FAV.' F
                                  WHERE I.status = :status
                                    AND I.deleted = 0
                                    AND I.id = F.item_id
                                    AND F.user_id = :user
                                    AND I.cat_id = C.id
                                    '.$this->db->langAnd(true, 'C', 'CL').'
                                  ORDER BY I.publicated_order', $aBind);
    }

    /**
     * Список объявлений пользователя в профиле (фронтенд)
     * @param integer $nUserID ID пользователя
     * @param bool $bCount только подсчет кол-ва объявлений
     * @param string $sqlLimit
     * @return array
     */
    function itemsListProfile($nUserID, $bCount = false, $sqlLimit = '') # frontend
    {
        $aBind = array(':user'=>$nUserID);

        if ($bCount) {
            return $this->db->one_data('SELECT COUNT(I.id) FROM '.TABLE_BBS_ITEMS.' I WHERE I.user_id = :user', $aBind );
        }

        return $this->db->select('SELECT I.id, I.link, I.title, I.status, I.status_prev, I.moderated,
                                    I.blocked_num, I.blocked_reason, I.publicated_to,
                                    I.imgcnt, I.img_m, I.price, I.price_curr, I.price_params, I.addr_full,
                                    I.svc, (I.svc & '.BBS::SERVICE_FIX.') as fixed, (I.svc & '.BBS::SERVICE_MARK.') as marked,
                                    I.press_status, I.press_date,
                                    CL.title as cat_title, C.prices as cat_prices
                                  FROM '.TABLE_BBS_ITEMS.' I,
                                       '.TABLE_BBS_CATEGORIES.' C,
                                       '.TABLE_BBS_CATEGORIES_LANG.' CL
                                  WHERE I.user_id = :user
                                    AND I.status != '.BBS::STATUS_NOTACTIVATED.'
                                    AND I.deleted = 0
                                    AND I.cat_id = C.id
                                    '.$this->db->langAnd(true, 'C', 'CL').'
                                  ORDER BY fixed DESC, I.fixed_order DESC, I.publicated_order DESC '.
                                  $sqlLimit, $aBind);
    }

    /**
     * Список объявлений компании (фронтенд)
     * @param integer $nCompanyID ID компании
     * @param bool $bCount только подсчет кол-ва объявлений
     * @param string $sqlLimit
     * @return array
     */
    function itemsListCompany($nCompanyID, $bCount = false, $sqlLimit = '') # frontend
    {
        $sqlFilter = array(
            'I.company_id = :company',
            'I.status = '.BBS::STATUS_PUBLICATED,
            'I.deleted = 0',
        );

        if(BBS::premoderation()){
            $sqlFilter[] = 'I.moderated > 0';
        }

        $aBind = array(':company'=>$nCompanyID);

        if ($bCount) {
            return $this->db->one_data('SELECT COUNT(I.id) FROM '.TABLE_BBS_ITEMS.' I WHERE '.join(' AND ', $sqlFilter), $aBind );
        }

        return $this->db->select('SELECT I.id, I.link, I.title, I.status, I.status_prev, I.moderated,
                                    I.blocked_num, I.blocked_reason, I.publicated_to,
                                    I.imgcnt, I.img_m, I.price, I.price_curr, I.price_params, I.addr_full,
                                    I.svc, (I.svc & '.BBS::SERVICE_FIX.') as fixed, (I.svc & '.BBS::SERVICE_MARK.') as marked,
                                    I.press_status, I.press_date,
                                    CL.title as cat_title, C.prices as cat_prices
                                  FROM '.TABLE_BBS_ITEMS.' I,
                                       '.TABLE_BBS_CATEGORIES.' C,
                                       '.TABLE_BBS_CATEGORIES_LANG.' CL
                                WHERE I.cat_id = C.id '.$this->db->langAnd(true, 'C', 'CL').' AND '.join(' AND ', $sqlFilter).'
                                ORDER BY fixed DESC, I.fixed_order DESC, I.publicated_order DESC
                            '.$sqlLimit, $aBind);
    }

    function itemsListVIP($nCatID = 0, $nTypeID = 0, $nCityID = 0)
    {
        $isServiceOn = $this->controller->svcIsVIPEnabled($aVipSettings);
        if( ! $isServiceOn ) return array();

        $sqlFilter = array('(I.svc & '.BBS::SERVICE_VIP.')', 'I.vip_to >= :now', 'I.status = '.BBS::STATUS_PUBLICATED);
        if( false ) {
            if( $nCatID > 0 ) { # категория
                $sqlFilter[] = 'I.cat_id = '.$nCatID;
            }
            if($nTypeID > 0) { # тип
                $sqlFilter[] = 'I.type_id = '.$nTypeID;
            }
            if($nCityID > 0) { # регион
                $sqlFilter[] = 'I.city_id = '.$nCityID;
            }
        }

        $aData = $this->db->select('SELECT I.id, I.link, I.title, I.price, I.price_curr, I.price_params,I.imgcnt, I.imgfav
                FROM '.TABLE_BBS_ITEMS.' I
                WHERE '.join(' AND ', $sqlFilter).'
                ORDER BY RAND()
                '. $this->db->prepareLimit(0, $aVipSettings['cnt']),
            array(':now'=>$this->db->now()));
        if( empty($aData) ) $aData = array();
        return $aData;
    }


    /**
     * Получение данных объявления
     * @param integer $nItemID ID объявления
     * @param array|string $aFields требуемые поля объявления
     * @param array|string $aUserFields требуемые данные о пользователе (авторе объявления)
     * @return array
     */
    function itemData($nItemID, $aFields = '*', $aUserFields = array())
    {
        if (empty($nItemID)) return array();

        $aParams = array();
        # item params
        if (empty($aFields)) $aFields = '*';
        if (!is_array($aFields)) $aFields = array($aFields);
        foreach ($aFields as $v) {
            $aParams[] = 'I.'.$v;
        }
        # user params
        $bJoinUsers = false;
        if ( ! empty($aUserFields)) {
            if (is_string($aUserFields)) $aUserFields = array($aUserFields);
            foreach ($aUserFields as $v) {
                $aParams[] = 'U.'.$v;
            }
            $bJoinUsers = true;
        }

        return $this->db->one_array('SELECT '.join(',', $aParams).'
                                  FROM '.TABLE_BBS_ITEMS.' I
                                       '.($bJoinUsers ? ', '.TABLE_USERS.' U' : '').'
                                  WHERE I.id = :id '.($bJoinUsers ? 'AND I.user_id = U.user_id' : ''),
                                  array(':id'=>$nItemID));
    }

    /**
     * Получение данных объявления при редактировании
     * @param integer $nItemID ID объявления
     * @return array
     */
    function itemDataEdit($nItemID)
    {
        return $this->itemData($nItemID,
            array('*', 'title_edit as title'),
            array('email as user_email', 'blocked as user_blocked'));
    }

    /**
     * Получение данных объявления для просмотра (фронтенд)
     * @param integer $nItemID ID объявления
     * @return array
     */
    function itemDataView($nItemID)
    {
        $aData = $this->db->one_array('SELECT I.*,
                                    CL.title as cat_title, C.addr as cat_addr, C.prices as cat_prices,
                                    T.title_'.LNG.' as cat_type_title, I.company_id
                                  FROM '.TABLE_BBS_ITEMS.' I
                                  LEFT JOIN '.TABLE_BBS_CATEGORIES_TYPES.' T ON I.cat_type = T.id,
                                       '.TABLE_BBS_CATEGORIES.' C,
                                       '.TABLE_BBS_CATEGORIES_LANG.' CL
                                  WHERE I.id = :id
                                    AND I.cat_id = C.id '.$this->db->langAnd(true, 'C', 'CL'),
                                  array(':id'=>$nItemID));
        if (empty($aData)) {
            return false;
        }

        # Находится ли ОБ в избранном у пользователя
        $aData['fav'] = false;
        if ($userID = User::id()) {
            $nFavCnt = $this->security->userCounter('bbs_fav');
            if ($nFavCnt > 0) {
                $res = $this->db->one_data('SELECT item_id FROM '.TABLE_BBS_ITEMS_FAV.'
                    WHERE user_id = :user AND item_id = :item', array(
                        ':user' => $userID, ':item' => $nItemID
                    ));
                $aData['fav'] = ! empty($res);
            }
        }

        # Получаем данные о компании
        if ($aData['company_id'] > 0) {
            $aData['company_data'] = Items::model()->companyData($aData['company_id']);
        }

        return $aData;
    }

    /**
     * Получение данных объявления для отправки email-уведомления
     * @param integer $nItemID ID объявления
     * @return array|bool
     */
    function itemData2Email($nItemID)
    {
        $aData = $this->itemData($nItemID,
            array('id as item_id', 'status', 'deleted', 'link as item_link',
                  'title as item_title'),
            array('user_id', 'name', 'email', 'blocked as user_blocked'));

        do {
            if (empty($aData)) break;
            # ОБ удалялось пользователем
            if ( ! empty($aData['deleted'])) break;
            # ОБ неактивировано
            if ((int)$aData['status'] === BBS::STATUS_NOTACTIVATED) break;
            # Проверяем владельца:
            # - заблокирован
            if ( ! empty($aData['user_blocked'])) break;
            return $aData;
        } while(false);

        return false;
    }

    function itemsDataByFilter($aFilter, $aFields = '*', $sqlOrder = '', $nLimit = false)
    {
        $aFilter = $this->prepareFilter( $aFilter );

        $aParams = array();
        if (empty($aFields)) $aFields = '*';
        if ( ! is_array($aFields)) $aFields = array($aFields);
        foreach ($aFields as $v) {
            $aParams[] = $v;
        }

        return $this->db->select_key('SELECT '.join(',', $aParams).'
                                  FROM '.TABLE_BBS_ITEMS.'
                                  '.$aFilter['where']
                                  .(!empty($sqlOrder) ? ' ORDER BY '.$sqlOrder : '')
                                  .(!empty($nLimit) ? $this->db->prepareLimit(0,$nLimit) : ''),
                                  'id',
                                  $aFilter['bind']);
    }

    /**
     * Сохранение объявления
     * @param integer $nItemID ID объявления
     * @param array $aData данные объявления
     * @param mixed $sDynpropsDataKey ключ данных дин. свойств, например 'd' или FALSE
     * @return boolean|integer
     */
    function itemSave($nItemID, $aData, $sDynpropsDataKey = false)
    {
        if (empty($aData)) return false;
        if ( ! empty($sDynpropsDataKey) && ! empty($aData['cat_id'])) {
            $aDataDP = $this->controller->dpSave( $aData['cat_id'], $sDynpropsDataKey );
            $aData = array_merge($aData, $aDataDP);
        }

        if (isset($aData['price_params'])) {
            $aData['price_params'] = array_sum($aData['price_params']);
        }

        if ($nItemID > 0)
        {
            $aData['modified'] = $this->db->now(); # Дата изменения

            $res = $this->db->update(TABLE_BBS_ITEMS, $aData, array('id'=>$nItemID));
        }
        else
        {
            if ( ! isset($aData['user_id']) ) {
                $aData['user_id'] = $this->security->getUserID();
            }
            $aData['user_ip'] = Request::remoteAddress();
            $aData['created'] = $this->db->now(); # Дата создания
            $aData['modified'] = $this->db->now(); # Дата изменения

            $res = $nItemID = $this->db->insert(TABLE_BBS_ITEMS, $aData);
        }

        if (!$res) return $res;

        # формируем URL просмотра
        $aData = $this->itemData($nItemID, array('keyword', 'city_id'));
        $this->db->update(TABLE_BBS_ITEMS, array(
            'link' => BBS::urlView($nItemID, $aData['keyword'], $aData['city_id']),
        ), array('id'=>$nItemID));

        return $res;
    }

    /**
     * Обновляем данные о нескольких объявлениях
     * @param array $aItemsID ID объявлений
     * @param array $aData данные
     * @return integer кол-во обновленных объявлений
     */
    function itemsSave($aItemsID, $aData)
    {
        return $this->db->update(TABLE_BBS_ITEMS, $aData, array('id'=>$aItemsID));
    }

    /**
     * Обновление счетчиков в категориях/типах
     * @param array $aCatsID ID категории
     * @param array $aTypesID ID типов
     * @param bool $bIncrement накручиваем/откручиваем
     * @param bool $bOneItem
     */
    function itemsCounterUpdate($aCatsID, $aTypesID, $bIncrement = true, $bOneItem = false)
    {
        $act = ($bIncrement?'+':'-');

        if ( ! empty($aCatsID))
        {
            if ($bOneItem) {
                $this->db->exec('UPDATE '.TABLE_BBS_CATEGORIES.'
                    SET items = items '.$act.' 1
                    WHERE id IN('.join(',', $aCatsID).')
                ');
            } else {
                $aUpdateData = array();
                foreach ($aCatsID as $k=>$i){
                    $aUpdateData[] = "WHEN $k THEN (items $act $i)";
                }
                if ( ! empty($aUpdateData)) {
                    $this->db->exec('UPDATE '.TABLE_BBS_CATEGORIES.'
                                        SET items = CASE id '.join(' ', $aUpdateData).' ELSE items END
                                        WHERE id IN ('.join(',',array_keys($aCatsID)).')' );
                }
            }
        }
        if ( ! empty($aTypesID))
        {
            if ($bOneItem) {
                $this->db->exec('UPDATE '.TABLE_BBS_CATEGORIES_TYPES.'
                    SET items = items '.$act.' 1 WHERE id IN ('.join(',', $aTypesID).')
                ');
            } else {
                $aUpdateData = array();
                foreach ($aTypesID as $k=>$i){
                    $aUpdateData[] = "WHEN $k THEN (items $act $i)";
                }
                if ( ! empty($aUpdateData)) {
                    $this->db->exec('UPDATE '.TABLE_BBS_CATEGORIES_TYPES.'
                                        SET items = CASE id '.join(' ', $aUpdateData).' ELSE items END
                                        WHERE id IN ('.join(',',array_keys($aTypesID)).')' );
                }
            }
        }
    }

    /**
     * Переключатели объявления
     * @param integer $nItemID ID объявления
     * @param string $sField переключаемое поле
     * @return mixed @see toggleInt
     */
    function itemToggle($nItemID, $sField)
    {
        switch ($sField) {
            case '?': {
                //return $this->toggleInt(TABLE_BBS_ITEMS, $nItemID, $sField, 'id');
            } break;
        }
    }

    /**
     * Удаление объявления
     * @param integer $nItemID ID объявления
     * @return boolean
     */
    function itemDelete($nItemID)
    {
        if (empty($nItemID)) return false;
        $res = $this->db->delete(TABLE_BBS_ITEMS, array('id' => $nItemID));
        if ( ! empty($res)) {
            return true;
        }
        return false;
    }

    /**
     * Обновление счетчиков объявлений + снятие с публикации по крону
     * рекомендуемая периодичность: каждые 5 минут
     */
    function itemsCron()
    {


        # Удаляем неактивированные объявления по прошествии срока
        $aItemsID = $this->db->select_one_column('SELECT id
            FROM ' . TABLE_BBS_ITEMS . '
            WHERE status = :status
              AND activate_expire <= :now',
            array(
                ':status' => BBS::STATUS_NOTACTIVATED,
                ':now'    => $this->db->now()
            )
        );
        if ( ! empty($aItemsID)) {
            foreach($aItemsID as $v) {
                $this->controller->itemDelete($v);
                # email уведомления не отправляем, поскольку email адреса не подтверджались
            }
        }

        # Снимаем с публикации просроченные объявления
        $this->db->exec('UPDATE '.TABLE_BBS_ITEMS.'
                SET status = '.BBS::STATUS_PUBLICATED_OUT.',
                    status_prev = '.BBS::STATUS_PUBLICATED.'
                WHERE status = '.BBS::STATUS_PUBLICATED.'
                  AND publicated_to <= :now',
                  array(':now'=>$this->db->now()));

        # Выполняем пересчет счетчиков ОБ в категориях:
        $this->db->exec('UPDATE '.TABLE_BBS_CATEGORIES.' SET items = 0');
        for ($i=1;$i<=3;$i++) {
            $this->db->exec('UPDATE '.TABLE_BBS_CATEGORIES.' C,
                    (SELECT I.cat_id'.$i.' as id, COUNT(I.id) as items
                        FROM '.TABLE_BBS_ITEMS.' I
                        WHERE I.status = '.BBS::STATUS_PUBLICATED.'
                          AND I.deleted = 0
                          AND I.cat_id'.$i.' != 0
                          '.(BBS::premoderation() ? ' AND I.moderated > 0' : '').'
                     GROUP BY I.cat_id'.$i.') as X
                SET C.items = X.items
                WHERE C.numlevel = '.$i.' AND C.id = X.id
            ');
        }

        # Выполняем пересчет счетчиков ОБ в типах категорий:
        $this->db->exec('UPDATE '.TABLE_BBS_CATEGORIES_TYPES.' SET items = 0');
        $this->db->exec('UPDATE '.TABLE_BBS_CATEGORIES_TYPES.' T,
                 (SELECT I.cat_type as id, COUNT(I.id) as items
                    FROM '.TABLE_BBS_ITEMS.' I
                    WHERE I.status = '.BBS::STATUS_PUBLICATED.'
                      AND I.deleted = 0
                      AND I.cat_type != 0
                      '.(BBS::premoderation() ? ' AND I.moderated > 0' : '').'
                 GROUP BY I.cat_type) as X
            SET T.items = X.items
            WHERE T.id = X.id
        ');
    }

    /**
     * Перестраивание ссылок во всех записях
     */
    public function itemsLinksRebuild()
    {
        $model = $this;
        $this->db->select_rows_chunked(TABLE_BBS_ITEMS, array('id','keyword','city_id'),
                                       array(),'id',array(), function ($items) use ($model) {
            foreach ($items as $v) {
                $model->db->update(TABLE_BBS_ITEMS, array(
                    'link' => BBS::urlView($v['id'], $v['keyword'], $v['city_id'])
                ), array('id'=>$v['id']));
            }
        }, 100);
    }

    # --------------------------------------------------------------------
    # Категории

    /**
     * Список категорий
     * @param array $aFilter фильтр списка категорий
     * @param bool $bCount только подсчет кол-ва категорий
     * @param array $aBind подстановочные данные
     * @param string $sqlLimit
     * @param string $sqlOrder
     * @return mixed
     */
    function categoriesListing($aFilter, $bCount = false, $aBind = array(), $sqlLimit = '', $sqlOrder = 'numleft') //admin
    {
        $aFilter[':lang'] = $this->db->langAnd(false, 'C', 'CL');
        $aFilter[] = 'pid != 0';
        $aFilter = $this->prepareFilter( $aFilter, 'C', $aBind );

        if ($bCount) {
            return $this->db->one_data('SELECT COUNT(C.id) FROM '.TABLE_BBS_CATEGORIES.' C, '.TABLE_BBS_CATEGORIES_LANG.' CL '.$aFilter['where'], $aFilter['bind'] );
        }

        return $this->db->select('SELECT C.id, C.created, CL.title, CL.mtitle, C.addr, C.prices, C.enabled, C.pid, C.numlevel, ((C.numright-C.numleft)-1) as node
               FROM '.TABLE_BBS_CATEGORIES.' C, '.TABLE_BBS_CATEGORIES_LANG.' CL
               '.$aFilter['where']
        .( ! empty($sqlOrder) ? ' ORDER BY '.$sqlOrder : '')
        .$sqlLimit, $aFilter['bind'] );
    }

    /**
     * Список категорий (фронтенд)
     * @param integer $nNumlevelMax максимальный уровень вложенности категорий
     * @param boolean $bRegionsFilter учитывать фильтрацию по региону
     * @return mixed
     */
    function categoriesList($nNumlevelMax = 1, $bRegionsFilter = false) # frontend
    {
        $aFilter[':lang'] = $this->db->langAnd(false, 'C', 'CL');
        $aFilter[] = 'pid != 0';
        $aFilter[] = 'enabled = 1';
        if ( ! empty($nNumlevelMax) ) {
            $aFilter[] = 'numlevel <= '.$nNumlevelMax;
        }
        $aFilter[':pipEnabled'] = 'CP.enabled = 1';

        $aFilter = $this->prepareFilter($aFilter, 'C');

        if ($bRegionsFilter)
        {
            # подсчет ОБ в категории считается только для основных категорий
            return $this->db->select('SELECT C.id, C.pid, CL.title, COUNT(I.id) as items, C.keyword
                                      FROM '.TABLE_BBS_CATEGORIES.' C
                                        LEFT JOIN '.TABLE_BBS_CATEGORIES.' CP ON C.pid = CP.id
                                        LEFT JOIN '.TABLE_BBS_ITEMS.' I ON I.cat_id1 = C.id
                                            AND I.status = '.BBS::STATUS_PUBLICATED.'
                                            AND I.deleted = 0
                                            AND I.city_id = '.Geo::cityID().'
                                        , '.TABLE_BBS_CATEGORIES_LANG.' CL
                                      '.$aFilter['where'].'
                                      GROUP BY C.id
                                      ORDER BY C.numleft',
                                      $aFilter['bind']);
        } else {
            return $this->db->select('SELECT C.id, C.pid, CL.title, C.items, C.keyword
                                      FROM '.TABLE_BBS_CATEGORIES.' C
                                        LEFT JOIN '.TABLE_BBS_CATEGORIES.' CP ON C.pid = CP.id
                                        , '.TABLE_BBS_CATEGORIES_LANG.' CL
                                      '.$aFilter['where'].'
                                      ORDER BY C.numleft',
                                      $aFilter['bind']);
        }
    }

    /**
     * Получение данных категории
     * @param integer $nCategoryID ID категории
     * @param boolean $bEdit при редактировании
     * @return array
     */
    function categoryData($nCategoryID, $bEdit = false)
    {
        if ($bEdit) {
            $aData = $this->db->one_array('SELECT C.*, ((C.numright-C.numleft)-1) as node
                    FROM '.TABLE_BBS_CATEGORIES.' C
                    WHERE C.id = :id',
                    array(':id'=>$nCategoryID));
            if ( ! empty($aData)) {
                $this->db->langSelect($nCategoryID, $aData, $this->langCategories, TABLE_BBS_CATEGORIES_LANG);
            }
        } else {
            $aData = $this->db->one_array('SELECT C.id, C.numlevel, CL.title, C.keyword,
                        C.prices, C.addr, ((C.numright-C.numleft)-1) as subs,
                        CL.mtitle,
                        CL.mkeywords,
                        CL.mdescription
                    FROM '.TABLE_BBS_CATEGORIES.' C
                       , '.TABLE_BBS_CATEGORIES_LANG.' CL
                    WHERE C.id = :id '.$this->db->langAnd(true, 'C', 'CL'),
                    array(':id'=>$nCategoryID));
        }
        return $aData;
    }

    function categoryDataSearch($sKeyword)
    {
        return $this->db->one_array('SELECT C.id, C.pid, CL.title, C.numlevel as lvl,
                        C.prices, C.addr, ((C.numright-C.numleft)-1) as subs, C.keyword,
                        CL.titleh1, CL.mtitle, CL.mkeywords, CL.mdescription, C.mtemplate
                    FROM '.TABLE_BBS_CATEGORIES.' C
                       , '.TABLE_BBS_CATEGORIES_LANG.' CL
                    WHERE C.keyword = :key AND C.enabled = 1 '.$this->db->langAnd(true, 'C', 'CL'),
                    array(':key'=>$sKeyword));
    }

    /**
     * Сохранение категории
     * @param integer $nCategoryID ID категории
     * @param array $aData данные категории
     * @return boolean|integer
     */
    function categorySave($nCategoryID, $aDataAll)
    {
        if (empty($aDataAll)) return false;

        $aData = array_diff_key($aDataAll, $this->langCategories);
        $aData['title'] = $aDataAll['title'][LNG];

        if ($nCategoryID > 0)
        {
            $aData['modified'] = $this->db->now(); # Дата изменения
            if (isset($aData['pid'])) unset($aData['pid']); # запрет изменения pid

            $res = $this->db->update(TABLE_BBS_CATEGORIES, $aData, array('id'=>$nCategoryID));
            $this->db->langUpdate($nCategoryID, $aDataAll, $this->langCategories, TABLE_BBS_CATEGORIES_LANG);
            return !empty($res);
        }
        else
        {
            $aData['created'] = $this->db->now(); # Дата создания
            $aData['modified'] = $this->db->now(); # Дата изменения

            $nCategoryID = $this->treeCategories->insertNode($aData['pid']);
            if ($nCategoryID > 0) {
                unset($aData['pid']);
                $this->db->update(TABLE_BBS_CATEGORIES, $aData, array('id'=>$nCategoryID));
                $this->db->langInsert($nCategoryID, $aDataAll, $this->langCategories, TABLE_BBS_CATEGORIES_LANG);
            }
            return $nCategoryID;
        }
    }

    /**
     * Переключатели категории
     * @param integer $nCategoryID ID категории
     * @param string $sField переключаемое поле
     * @return mixed @see toggleInt
     */
    function categoryToggle($nCategoryID, $sField)
    {
        switch ($sField) {
            case 'enabled': { # Включена
                return $this->toggleInt(TABLE_BBS_CATEGORIES, $nCategoryID, $sField, 'id');
            } break;
            case 'prices': { # Цена
                return $this->toggleInt(TABLE_BBS_CATEGORIES, $nCategoryID, $sField, 'id');
            } break;
            case 'addr': { # Точка на карте
                return $this->toggleInt(TABLE_BBS_CATEGORIES, $nCategoryID, $sField, 'id');
            } break;
        }
    }

    /**
     * Перемещение категории
     * @return mixed @see rotateTablednd
     */
    function categoriesRotate()
    {
        return $this->treeCategories->rotateTablednd();
    }

    /**
     * Удаление категории
     * @param integer $nCategoryID ID категории
     * @return boolean
     */
    function categoryDelete($nCategoryID)
    {
        if (empty($nCategoryID)) return false;
        $nSubCnt = $this->categorySubCount($nCategoryID);
        if ( ! empty($nSubCnt)) {
            $this->errors->set(_t('bbs','Невозможно выполнить удаление категории при наличии подкатегорий'));
            return false;
        }

        $nItems =  $this->db->one_data('SELECT COUNT(I.id) FROM '.TABLE_BBS_ITEMS.' I WHERE I.cat_id = :id', array(':id'=>$nCategoryID));
        if ( ! empty($nItems)) {
            $this->errors->set(_t('bbs','Невозможно выполнить удаление категории при наличии вложенных элементов'));
            return false;
        }

        $aDeletedID = $this->treeCategories->deleteNode($nCategoryID);
        $res = ! empty($aDeletedID);
        if ( ! empty($res)) {
            $this->db->delete(TABLE_BBS_CATEGORIES_LANG, array('id'=>$nCategoryID));
            return true;
        } else {
            $this->errors->set(_t('bbs','Ошибка удаления категории'));
        }
        return false;
    }

    /**
     * Получаем кол-во вложенных категорий
     */
    function categorySubCount($nCategoryID)
    {
        return $this->treeCategories->getChildrenCount($nCategoryID);
    }

    /**
     * Формирование списка подкатегорий
     * @param integer $nCategoryID ID категории
     * @param mixed $mOptions формировать select-options или FALSE
     * @param bool $bEnabled только включенные
     * @return array|string
     */
    function categorySubOptions($nCategoryID, $mOptions = false, $bEnabled = true)
    {
        $aData = $this->db->select('SELECT C.id, CL.title
                    FROM '.TABLE_BBS_CATEGORIES.' C, '.TABLE_BBS_CATEGORIES_LANG.' CL
                    WHERE C.pid = :pid '.($bEnabled ? ' AND C.enabled = 1' : '').'
                      '.$this->db->langAnd(true, 'C', 'CL').'
                    ORDER BY C.numleft', array(':pid'=>$nCategoryID));

        if (empty($mOptions)) return $aData;

        return HTML::selectOptions( $aData, $mOptions['sel'], $mOptions['empty'], 'id', 'title' );
    }


    /**
     * Формирование списка основных категорий
     * @param integer $nSelectedID ID выбранной категории
     * @param mixed $mEmptyOpt невыбранное значение
     * @param integer $nType тип списка: 0 - все(кроме корневого), 1 - список при добавлении категории, 2 - список при добавлении записи
     * @param array $aOnlyID только список определенных категорий
     * @return string <option></option>...
     */
    function categoriesOptions($nSelectedID = 0, $mEmptyOpt = false, $nType = 0, $aOnlyID = array())
    {
        $aFilter = array();
        if ($nType == 1) {
            $aFilter[] = 'numlevel < 3';
        } else {
            $aFilter[] = 'numlevel > 0';
        }
        if ( ! empty($aOnlyID)) {
            $aFilter[':only'] = '(C.id IN ('.join(',', $aOnlyID).') OR C.pid IN('.join(',', $aOnlyID).'))';
        }

        # Chrome не понимает style="padding" в option
        $bUsePadding = ( mb_stripos(Request::userAgent(), 'chrome')===false );
        $bJoinItems = ( $nType > 0 );
        $aFilter[':lang'] = $this->db->langAnd(false, 'C', 'CL');
        $aFilter = $this->prepareFilter( $aFilter, 'C' );

        $aCategories = $this->db->select('SELECT C.id, CL.title, C.numlevel, ((C.numright-C.numleft)-1) as node
                    '.( $bJoinItems ? ', COUNT(I.id) as items ' : '' ).'
               FROM '.TABLE_BBS_CATEGORIES.' C
                    '.( $bJoinItems ? ' LEFT JOIN '.TABLE_BBS_ITEMS.' I ON C.id = I.cat_id ' : '').'
                    , '.TABLE_BBS_CATEGORIES_LANG.' CL
               '.$aFilter['where'].'
               GROUP BY C.id
               ORDER BY C.numleft', $aFilter['bind'] );

        $sOptions = '';
        foreach ($aCategories as $v) {
            $nNumlevel = &$v['numlevel'];
            $bDisable = ( $nType>0 && ($nType == 2 ? $v['node'] > 0 : ( $nNumlevel>0 && $v['items']>0 ) ) );
            $sOptions .= '<option value="'.$v['id'].'" '.
                ($bUsePadding && $nNumlevel>1 ? 'style="padding-left:'.($nNumlevel*10).'px;" ':'').
                ($v['id'] == $nSelectedID?' selected':'').
                ($bDisable ? ' disabled' : '').
                '>'.( ! $bUsePadding && $nNumlevel>1 ? str_repeat('  ', $nNumlevel) :'').$v['title'].'</option>';
        }

        if ($mEmptyOpt!==false) {
            $nValue = 0;
            if (is_array($mEmptyOpt)) {
                $nValue = key($mEmptyOpt);
                $mEmptyOpt = current($mEmptyOpt);
            }
            $sOptions = '<option value="'.$nValue.'" class="bold">'.$mEmptyOpt.'</option>'.$sOptions;
        }

        return $sOptions;
    }


    /**
     * Формирование списков категорий (при добавлении/редактировании записи)
     * @param array $aCategoriesID ID категорий [lvl=>selectedID, ...]
     * @param mixed $mOptions формировать select-options или нет (false)
     * @param bool $bEdit для формы редактирования
     * @return array [lvl=>[a=>selectedID, categories=>список категорий(массив или options)],...]
     */
    function categoriesOptionsByLevel($aCategoriesID, $mOptions = false, $bEditForm = false)
    {
        if (empty($aCategoriesID)) return array();

        # формируем список требуемых уровней категорий
        $aLevels = array(); $bFill = true; $parentID = 1;
        foreach ($aCategoriesID as $lvl=>$nCategoryID) {
            if ($nCategoryID || $bFill) {
                $aLevels[$lvl] = $parentID;
                if ( ! $nCategoryID) break;
                $parentID = $nCategoryID;
            } else {
                break;
            }
        }

        if (empty($aLevels)) return array();

        $aData = $this->db->select('SELECT C.id, CL.title, C.numlevel as lvl, C.keyword
                    FROM '.TABLE_BBS_CATEGORIES.' C, '.TABLE_BBS_CATEGORIES_LANG.' CL
                    WHERE C.numlevel IN ('.join(',', array_keys($aLevels)).')
                      AND C.pid IN('.join(',', $aLevels).')
                      AND '.( $bEditForm ? '( C.id IN('.join(',', $aCategoriesID).') OR C.enabled = 1 )' :
                      ' C.enabled = 1 ').'
                      '.$this->db->langAnd(true, 'C', 'CL').'
                    ORDER BY C.numleft');
        if (empty($aData)) return array();

        $aLevels = array();
        foreach ($aData as $v) {
            $aLevels[$v['lvl']][$v['id']] = $v;
        } unset($aData);

        foreach ($aCategoriesID as $lvl=>$nSelectedID)
        {
            if (isset($aLevels[$lvl])) {
                $aCategoriesID[$lvl] = array(
                    'a' => $nSelectedID,
                    'categories' => ( !empty($mOptions) ?
                        HTML::selectOptions($aLevels[$lvl], $nSelectedID, $mOptions['empty'], 'id', 'title') :
                        $aLevels[$lvl] ),
                );
            } else {
                $aCategoriesID[$lvl] = array(
                    'a' => $nSelectedID,
                    'categories' => false,
                );
            }
        }
        return $aCategoriesID;
    }


    /**
     * Получаем данные parent-категорий
     * @param integer $nCategoryID ID категории
     * @param bool $bIncludingSelf включать текущую в итоговых список
     * @param bool $bExludeRoot исключить корневой раздел
     * @return array array(lvl=>id, ...)
     */
    function categoryParentsID($nCategoryID, $bIncludingSelf = true, $bExludeRoot = true)
    {
        if (empty($nCategoryID)) return array(1=>0);
        $aData = $this->treeCategories->getNodeParentsID($nCategoryID, ($bExludeRoot ? ' AND numlevel > 0' : ''), $bIncludingSelf, array('id','numlevel'));
        $aParentsID = array();
        if ( ! empty($aData)) {
            foreach ($aData as $v) {
                $aParentsID[ $v['numlevel'] ] = $v['id'];
            }
        }
        return $aParentsID;
    }

    /**
     * Получаем названия parent-категорий
     * @param integer $nCategoryID ID категории
     * @param boolean $bIncludingSelf включать текущую в итоговых список
     * @param boolean $bExludeRoot исключить корневой раздел
     * @param mixed $mSeparator объединить в одну строку или FALSE
     * @return array array(lvl=>id, ...)
     */
    function categoryParentsTitle($nCategoryID, $bIncludingSelf = false, $bExludeRoot = false, $mSeparator = true)
    {
        $aParentsID = $this->treeCategories->getNodeParentsID($nCategoryID, ($bExludeRoot ? ' AND numlevel > 0' : ''), $bIncludingSelf);
        if (empty($aParentsID)) return ( $mSeparator!==false ? '' : array() );

        $aData = $this->db->select_one_column('SELECT CL.title
                   FROM '.TABLE_BBS_CATEGORIES.' C, '.TABLE_BBS_CATEGORIES_LANG.' CL
                   WHERE '.$this->db->prepareIN('C.id', $aParentsID).' '.$this->db->langAnd(true, 'C', 'CL').'
                   ORDER BY C.numleft');

        $aData = (!empty($aData) ? $aData : array());

        if ($mSeparator!==false) {
            return join('  '.($mSeparator===true ? '>' : $mSeparator).'  ', $aData);
        } else {
            return $aData;
        }
    }


    /**
     * Получаем данные о parent-категориях
     * @param integer $nCategoryID ID категории
     * @param boolean $bIncludingSelf включать текущую в итоговых список
     * @param boolean $bExludeRoot исключить корневой раздел
     * @return array array(lvl=>id, ...)
     */
    function categoryParentsData($nCategoryID, $bIncludingSelf = false, $bExludeRoot = false)
    {
        $aParentsID = $this->treeCategories->getNodeParentsID($nCategoryID, ($bExludeRoot ? ' AND numlevel > 0' : ''), $bIncludingSelf);
        if (empty($aParentsID)) return array();

        $aData = $this->db->select_key('SELECT C.id, C.keyword, CL.title
                   FROM '.TABLE_BBS_CATEGORIES.' C, '.TABLE_BBS_CATEGORIES_LANG.' CL
                   WHERE '.$this->db->prepareIN('C.id', $aParentsID).' '.$this->db->langAnd(true, 'C', 'CL').'
                   ORDER BY C.numleft', 'id');

        return (!empty($aData) ? $aData : array());
    }

    # ----------------------------------------------------------------
    # Типы категорий объявлений

    function cattypesListing($aFilter)
    {
        if (empty($aFilter)) {
            $aFilter = array();
        }

        $aFilter[':lang'] = $this->db->langAnd(false, 'C', 'CL');
        $aFilter[] = 'T.cat_id = C.id';

        $aFilter = $this->prepareFilter( $aFilter );

        return $this->db->select('SELECT T.*, T.title_'.LNG.' as title, CL.title as cat_title
                    FROM '.TABLE_BBS_CATEGORIES_TYPES.' T, '.TABLE_BBS_CATEGORIES.' C, '.TABLE_BBS_CATEGORIES_LANG.' CL
                    '.$aFilter['where'].'
                    ORDER BY C.numleft, T.num ASC', $aFilter['bind']);
    }

    function cattypeData($nTypeID, $aFields = array(), $bEdit = false)
    {
        if (empty($aFields)) $aFields = '*';

        $aParams = array();
        if ( ! is_array($aFields)) $aFields = array($aFields);
        foreach ($aFields as $v) {
            $aParams[] = $v;
        }

        $aData = $this->db->one_array('SELECT '.join(',', $aParams).'
                       FROM '.TABLE_BBS_CATEGORIES_TYPES.'
                       WHERE id = :id
                       LIMIT 1', array(':id'=>$nTypeID));

        if ($bEdit) {
            $this->db->langFieldsSelect($aData, $this->langCategoriesTypes);
        }

        return $aData;
    }

    function cattypeSave($nTypeID, $nCategoryID, $aData)
    {
        if ($nTypeID) {
            $aData['modified'] = $this->db->now();
            $this->db->langFieldsModify($aData, $this->langCategoriesTypes, $aData);
            return $this->db->update(TABLE_BBS_CATEGORIES_TYPES, $aData, array('id'=>$nTypeID));
        } else {
            $nNum = (integer)$this->db->one_data('SELECT MAX(num) FROM '.TABLE_BBS_CATEGORIES_TYPES.' WHERE cat_id = '.$nCategoryID);
            $aData['num'] = $nNum + 1;
            $aData['cat_id'] = $nCategoryID;
            $aData['created'] = $aData['modified'] = $this->db->now();
            $this->db->langFieldsModify($aData, $this->langCategoriesTypes, $aData);
            return $this->db->insert(TABLE_BBS_CATEGORIES_TYPES, $aData, 'id');
        }
    }

    function cattypeDelete($nTypeID)
    {
        if ( ! $nTypeID) return false;

        $nItems =  $this->db->one_data('SELECT COUNT(id) FROM '.TABLE_BBS_ITEMS.' WHERE cat_type = '.$nTypeID);
        if ( ! empty($nItems)) {
            $this->errors->set(_t('bbs','Невозможно удалить тип категории с объявлениями'));
            return false;
        }

        //удаляем только "свободный" тип
        $res = $this->db->exec('DELETE FROM '.TABLE_BBS_CATEGORIES_TYPES.' WHERE id = '.$nTypeID);
        return !empty($res);
    }

    function cattypeToggle($nTypeID, $sField)
    {
        if ( ! $nTypeID) return false;

        switch ($sField)
        {
            case 'enabled': {
                return $this->toggleInt(TABLE_BBS_CATEGORIES_TYPES, $nTypeID, 'enabled', 'id');
            } break;
        }
        return false;
    }

    function cattypesRotate($nCategoryID)
    {
        return $this->db->rotateTablednd(TABLE_BBS_CATEGORIES_TYPES, ' AND cat_id = '.$nCategoryID);
    }

    /**
     * Формирование списка типов, привязанных к категории
     * @param integer $nCategoryID ID категории
     * @param mixed $mOptions формировать select-options или FALSE
     * @return array|string
     */
    function cattypesByCategory($nCategoryID, $mOptions = false)
    {
        if (empty($nCategoryID)) {
            return array();
        }

        $aCategoryParentsID = $this->categoryParentsID($nCategoryID, true, true);
        $sQuery = 'SELECT T.id, T.title_'.LNG.' as title
            FROM '.TABLE_BBS_CATEGORIES_TYPES.' T, '.TABLE_BBS_CATEGORIES.' C
            WHERE T.cat_id IN ('.join(',', $aCategoryParentsID).') AND T.cat_id = C.id
            ORDER BY C.numleft, T.num ASC';

        $aTypes = $this->db->select($sQuery);

        if (empty($mOptions)) return $aTypes;

        return HTML::selectOptions( $aTypes, $mOptions['sel'], $mOptions['empty'], 'id', 'title' );
    }


    # ----------------------------------------------------------------
    # Жалобы

    function claimsListing($aFilter, $bCount = false, $sqlLimit = '')
    {
        $aFilter = $this->prepareFilter( $aFilter, 'CL' );

        if ($bCount) {
            return $this->db->one_data('SELECT COUNT(CL.id)
                                FROM '.TABLE_BBS_ITEMS_CLAIMS.' CL
                                '.$aFilter['where'], $aFilter['bind']);
        }

        $aData = $this->db->select('SELECT CL.*, U.name, U.blocked as ublocked, U.deleted as udeleted
                                FROM '.TABLE_BBS_ITEMS_CLAIMS.' CL
                                    LEFT JOIN '.TABLE_USERS.' U ON CL.user_id = U.user_id
                                '.$aFilter['where'].'
                            ORDER BY CL.created DESC'.$sqlLimit, $aFilter['bind']);

        if ( ! empty($aData) ) {
            $reasons = $this->controller->getItemClaimReasons();
            if ( ! empty($reasons))
            {
                foreach ($aData as $k=>$v) {
                    $r = $v['reasons'];
                    if ($r==0) continue;

                    $r_text = array();
                    foreach ($reasons as $rk=>$rv) {
                        if ($rk!=32 && $rk & $r) {
                            $r_text[] = $rv;
                        }
                    }
                    $aData[$k]['reasons_text'] = join(', ', $r_text);
                    $aData[$k]['other'] = $r & 32;
                }
            }
        }

        return $aData;
    }

    function claimData($nClaimID, $aFields = array())
    {
        if (empty($aFields)) $aFields = '*';

        $aParams = array();
        if ( ! is_array($aFields)) $aFields = array($aFields);
        foreach ($aFields as $v) {
            $aParams[] = $v;
        }

        return $this->db->one_array('SELECT '.join(',', $aParams).'
                       FROM '.TABLE_BBS_ITEMS_CLAIMS.'
                       WHERE id = :cid
                       LIMIT 1', array(':cid'=>$nClaimID));
    }

    function claimSave($nClaimID, $aData)
    {
        if ($nClaimID) {
            return $this->db->update(TABLE_BBS_ITEMS_CLAIMS, $aData, array('id'=>$nClaimID));
        } else {
            $aData['created'] = $this->db->now();
            $aData['user_id'] = $this->security->getUserID();
            $aData['ip'] = Request::remoteAddress();
            return $this->db->insert(TABLE_BBS_ITEMS_CLAIMS, $aData);
        }
    }

    function claimLastByUser($nUserID)
    {
        return $this->db->one_data('SELECT created
            FROM '.TABLE_BBS_ITEMS_CLAIMS.'
            WHERE user_id = :user
            ORDER BY created DESC LIMIT 1',
            array(':user'=>$nUserID));
    }

    function claimDelete($nClaimID)
    {
        if ( ! $nClaimID) return false;
        return $this->db->delete(TABLE_BBS_ITEMS_CLAIMS, $nClaimID);
    }

    function getLocaleTables()
    {
        return array(
            TABLE_BBS_CATEGORIES => array('type'=>'table','fields'=>$this->langCategories),
            TABLE_BBS_CATEGORIES_TYPES => array('type'=>'fields','fields'=>$this->langCategoriesTypes),
            TABLE_BBS_CATEGORIES_DYNPROPS => array('type'=>'fields','fields'=>array('title'=>TYPE_NOTAGS, 'description' => TYPE_NOTAGS)),
            TABLE_BBS_CATEGORIES_DYNPROPS_MULTI => array('type'=>'fields','fields'=>array('name'=>TYPE_NOTAGS)),
        );
    }
}