<?php

class Job extends JobBase
{
    /**
     * Роутинг
     */
    public function route()
    {
        $prefix = ( ! bff::subdomainsEnabled('job') ? 'job/' : '' );
        $res = bff::route(array(
            $prefix.'(resume|vacancy)/(.*)-([\d]+)\.html(.*)' => 'job/\\1_view/id=\\3', # /resume/1.html - просмотр резюме/вакансии
            $prefix.'(resume|vacancy)/search/(.*)(\/|)'       => 'job/search/stype=\\1&scat=\\2', # /resume/search/cat - поиск резюме/вакансии в категории
            $prefix.'(resume|vacancy)/(add|edit)(.*)'         => 'job/\\1_\\2',         # /resume/add    - добавление, редактирование, поиск резюме/вакансии
            $prefix.'(resume|vacancy)(.*)'                    => 'job/\\1_index',       # /resume/       - просмотр общего списка резюме/вакансий
            $prefix.'(fav|search|promote)(.*)'                => 'job/\\1',             # /fav/          - поиск, избранные резюме/вакансии
            $prefix.'company(.*)'                             => 'job/company',         # /company/      - список компаний
            $prefix.'agents(.*)'                              => 'job/agents',          # /agents/       - список "Кадровых агенств"
        ), true);

        if($res['event'] === false || ! method_exists($this, $res['event']))
            $res['event'] = 'index';

        return $this->$res['event']();
    }

    /**
     * Работа: главная
     */
    public function index()
    {
        $aData = $this->search_filter(0);
        
        $aData['latest_vacancy'] = $this->vacancy_minilist(5); 
        $aData['latest_resume']  = $this->resume_minilist(5); 

        $aData['company'] = $this->db->select('
                SELECT C.id, C.imgcnt, C.img_list as img, C.link
                FROM '.TABLE_JOB_VACANCY.' I, '.TABLE_ITEMS.' C
                WHERE I.company_id != 0
                  '.( $this->regionsFilterEnabled() ? ' AND I.city_id = '.Geo::cityID() : '' ).'
                  AND I.status = '.self::STATUS_PUBLICATED.'
                  '.(static::premoderation() ? ' AND I.moderated > 0 ' : '' ).'
                  AND I.company_id = C.id AND C.enabled = 1 AND C.imgcnt > 0
                GROUP BY C.id
                '.$this->db->prepareLimit(0,5));
        if( ! empty($aData['company'])) {
            foreach($aData['company'] as &$v) {
                $v['link'] = Items::url('view', $v['link']);
            } unset($v);
        }

        $aData['news'] = Publications::i()->indexCompaniesInCatNewsBlock(static::agentsCategory());

        # SEO: Главная страница
        $this->urlCorrection(static::url('index'));
        $this->seo()->canonicalUrl(static::url('index', array(), true));
        $this->setMeta('index', array(
            'region' => Geo::filter('title'),
        ), $aData);

        $this->initRightblock(__FUNCTION__, array('filter'=>&$aData['filter']));
        return $this->viewPHP($aData, 'index');
    }

    /**
     * Блок "Вакансии, Резюме" на главной
     */
    public function indexBlock()
    {
        $sql = array('I.status = '.self::STATUS_PUBLICATED,
                     'I.cat_id = C.id');
        if(static::premoderation()){
            $sql[] = 'I.moderated > 0';
        }


        $resume = $this->db->select('SELECT I.id, I.title, I.link, I.created, I.price, I.price_curr, I.price_torg,
                                    CL.title as cat_title
                                FROM '.TABLE_JOB_RESUME.' I,
                                     '.TABLE_JOB_CATEGORIES.' C,
                                     '.TABLE_JOB_CATEGORIES_LANG.' CL
                                WHERE '.join(' AND ', $sql).'
                                    '.( $this->regionsFilterEnabled() ? ' AND I.city_id = '.Geo::cityID() : '' ).
                                    $this->db->langAnd(true, 'C', 'CL').'
                                ORDER BY I.created DESC
                                '.$this->db->prepareLimit(0,4));
        if ( ! empty($resume)) {
            foreach ($resume as &$v) {
                $v['url'] = static::url('view', $v['link']);
            } unset($v);
            $resume = array_chunk($resume, 2);
            if (empty($resume[1])) $resume[] = array();
        } else {
            $resume = array(array(),array());
        }
        $aData['resume'] = &$resume;

        $vacancy = $this->db->select('SELECT I.id, I.title, I.link, I.created, I.price, I.price_curr, I.price_torg,
                                    CL.title as cat_title
                                FROM '.TABLE_JOB_VACANCY.' I,
                                     '.TABLE_JOB_CATEGORIES.' C,
                                     '.TABLE_JOB_CATEGORIES_LANG.' CL
                                WHERE '.join(' AND ', $sql).'
                                    '.( $this->regionsFilterEnabled() ? ' AND I.city_id = '.Geo::cityID() : '' ).
                                    $this->db->langAnd(true, 'C', 'CL').'
                                ORDER BY I.created DESC
                                '.$this->db->prepareLimit(0,4));
        if ( ! empty($vacancy)) {
            foreach ($vacancy as &$v) {
                $v['url'] = static::url('view', $v['link']);
            } unset($v);
            $vacancy = array_chunk($vacancy, 2);
            if (empty($vacancy[1])) $vacancy[] = array();
        } else {
            $vacancy = array(array(),array());
        }
        $aData['vacancy'] = &$vacancy;

        return $this->viewPHP($aData, 'index.block');
    }

    /**
     * Поиск вакансий или резюме
     */
    public function search()
    {
        $sType = $this->input->get('stype', TYPE_STR);
        switch ($sType) {
            case 'vacancy':
                list($sqlTable, $typeID, $nServiceMark, $nServiceFix) =
                    array(TABLE_JOB_VACANCY, self::TYPE_VACANCY, self::SERVICE_VACANCY_MARK, self::SERVICE_VACANCY_FIX); break;
            case 'resume':
                list($sqlTable, $typeID, $nServiceMark, $nServiceFix) =
                    array(TABLE_JOB_RESUME, self::TYPE_RESUME, self::SERVICE_RESUME_MARK, self::SERVICE_RESUME_FIX); break;
            default: $this->errors->error404();
        }

        $aData = $this->search_filter($typeID, true);
        $f = &$aData['f'];

        $nCatID = $f['cat'];
        $seoNoIndex = false;

        $sql = array();
        $sql[] = 'I.status = '.self::STATUS_PUBLICATED;
        if(static::premoderation()){
            $sql[] = 'I.moderated > 0';
        }
        $sql[] = 'I.price_curr = CURR.id';
        if ( ! Geo::cityIsOne() ) {
            if($f['rc'] > 0) { # город
                $sql[] = 'I.city_id = '.$f['rc'];
            }
            if (Geo::filterUrl('id') != $f['rc']) {
                $seoNoIndex = true;
            }
        }
        if($nCatID) { # проф. сфера
            $sql[] = 'I.cat_id = '.$nCatID;
        }
        $seoResetCounter = sizeof($sql) + 2; # всю фильтрацию ниже скрываем от индексации
        if($f['sh']) { # график
            $sql[] = '(I.schedule & '.$f['sh'].')';
        }
        # Зарплата (в основной валюте)
        if($f['pf'] > 0 || $f['pt'] > 0)
        {
            if($f['pf'] > 0) { # от
                $sql[] = '((I.price * CURR.rate) >= '.$f['pf'].' AND I.price_torg = 0 )';
                if($f['pt'] > 0 && $f['pt'] > $f['pf']) {
                    $sql[] = '((I.price * CURR.rate) <= '.$f['pt'].' AND I.price_torg = 0 )'; # если до, и до больше от
                }
            } else if($f['pt'] > 0) { # до
                $sql[] = '((I.price * CURR.rate) <= '.$f['pt'].' AND I.price_torg = 0 )';
            }
        }
        
        $joinCompany = ( $typeID == self::TYPE_VACANCY ?
            ' LEFT JOIN '.TABLE_ITEMS.' M ON M.id = I.company_id
              LEFT JOIN '.TABLE_ITEMS_LANG.' ML ON '.$this->db->langAnd(false, 'M', 'ML')
            : '' );
        
        if( ! empty($f['q'])) {
            $f['q'] = $this->input->cleanSearchString($f['q']);
            $sqlQ = $this->db->str2sql('%'.$f['q'].'%');
            $sql[] = '(I.title LIKE '.$sqlQ.' 
                OR I.company_title LIKE '.$sqlQ
                . ( ! empty($joinCompany) ? ' OR ML.title LIKE '.$sqlQ : '')
                . ')';
        }

        $nPerpage = 10;
        $nTotal = $this->db->one_data('SELECT COUNT(I.id) FROM '.$sqlTable.' I '.$joinCompany.', '.TABLE_CURRENCIES.' CURR WHERE '.join(' AND ', $sql));

        $sql[] = 'I.cat_id = C.id';
        $sql[] = 'I.city_id = R.id';

        $aRequestURI = parse_url( Request::uri() );
        $sQuery = ( ! empty($aRequestURI['query']) ? trim( preg_replace('/&?page=(\d+)/', '', $aRequestURI['query']), '&') : '' );
        $aData['pgn'] = $this->generatePagenationDots($nTotal, $nPerpage, 2, static::url('search', array('type'=>$f['type']))."?{$sQuery}&page={page}", $sqlLimit);
        $nPage = $this->input->get('page', TYPE_UINT); if (!$nPage) $nPage = 1;

        $nUserID = User::id();
        
        $aData['items'] = ( $nTotal > 0 ? $this->db->select('SELECT I.id, I.title, I.link, I.created, I.price, (I.price * CURR.rate) as price_cross, I.price_curr, I.price_torg,
                                    CL.title as cat_title, R.title_'.LNG.' as city_title, '.($nUserID ? ' (F.item_id)' : '0').' as fav,
                                    I.svc, (I.svc & '.$nServiceFix.') as fixed, (I.svc & '.$nServiceMark.') as marked,
                                    '.( $typeID == self::TYPE_VACANCY ?
                                        ' I.company_id, I.company_title, ML.title as item_title, M.link as item_link ' :
                                        ' I.fio, '.self::resumeYears ).'
                                FROM '.$sqlTable.' I
                                     '.$joinCompany.'
                                     '.($nUserID ? ' LEFT JOIN '.TABLE_JOB_FAV.' F ON I.id = F.item_id AND F.user_id = '.$nUserID.' AND F.type_id = '.$typeID : '').',
                                     '.TABLE_CURRENCIES.' CURR,
                                     '.TABLE_JOB_CATEGORIES.' C,
                                     '.TABLE_JOB_CATEGORIES_LANG.' CL,
                                     '.TABLE_REGIONS.' R
                                WHERE '.join(' AND ', $sql).$this->db->langAnd(true, 'C', 'CL').'
                                ORDER BY fixed DESC, I.fixed_order DESC, I.publicated_order DESC
                                '.$sqlLimit) : array() );
        # SEO:
        $aData['titleh1'] = '';
        $this->seo()->robotsIndex(!(sizeof($sql) - $seoResetCounter) && !$seoNoIndex);
        if ($nCatID > 0) {
            # SEO: Поиск вакансий / резюме в категории
            $aData['cat'] = $this->model->categoryData($nCatID, $typeID);
            $this->urlCorrection(static::url('search', array('type'=>$typeID, 'cat'=>$aData['cat']['keyword'])));
            $this->seo()->canonicalUrl(static::url('search', array('type'=>$typeID, 'cat'=>$aData['cat']['keyword']), true), array(
                'page' => $nPage
            ));
            $this->setMeta('search-category-'.$typeID, array(
                'region'   => Geo::filter('title'),
                'category' => $aData['cat']['title'],
                'page'     => $nPage,
            ), $aData['cat']);
            $aData['titleh1'] = $aData['cat']['titleh1'];
        } else {
            # SEO: Поиск вакансий / резюме
            $this->urlCorrection(static::url('search', array('type'=>$typeID)));
            $this->seo()->canonicalUrl(static::url('search', array('type'=>$typeID), true), array(
                'page' => $nPage
            ));
            $this->setMeta('search-'.$typeID, array(
                'region'   => Geo::filter('title'),
                'page'     => $nPage,
            ), $aData);
        }

        $this->initRightblock(__FUNCTION__, array('filter'=>&$aData['filter']));
        return $this->viewPHP($aData, $f['type_key'].'.search');
    }

    /**
     * Фильтр поиска
     * @param integer $typeID типа поиска
     * @param bool $bSubmit
     */
    protected function search_filter($typeID, $bSubmit = false)
    {
        $f = $this->input->postgetm(array(
            'q'  => TYPE_NOTAGS, # ключевые слова
            'rc' => TYPE_INT, # регион: город
            'cat' => TYPE_UINT, # проф. сфера
            'sh' => TYPE_ARRAY_UINT, # график
            'pf' => TYPE_PRICE, # цена, от
            'pt' => TYPE_PRICE, # цена, до
        ));

        $f['type'] = $typeID;
        $f['type_key'] = ($typeID == self::TYPE_RESUME ? 'resume' : 'vacancy');

        if( $f['rc'] === 0 ) {
            $f['rc'] = ( $this->regionsFilterEnabled() ? Geo::cityID() : -1 );
        }

        if( ! $f['cat']){
            $sCatKeyword = $this->input->get('scat', TYPE_NOTAGS);
            if($sCatKeyword){
                $aCategory = $this->model->categoryByKeyword($sCatKeyword, array('id'));
                if( ! empty($aCategory['id'])){
                    $f['cat'] = $aCategory['id'];
                }
            }
        }

        $aData = array('f'=>&$f);

        # проф. сфера
        $aData['cats'] = $this->categoriesOptions($f['cat'], _t('job','Не имеет значения'));

        # график
        $f['sh'] = array_sum($f['sh']);
        $aData['schedule'] = $this->getSchedule($f['sh'], true, 'sh');

        # активируем меню
        bff::setActiveMenu('//job/'.( ! $typeID ? 'all' : $f['type_key'] ), $bSubmit, false);
        return array(
            'filter' => $this->viewPHP($aData, 'search.filter'),
            'cats' => $aData['cats'],
            'f' => $f
        );
    }

    /**
     * Доп. ajax методы
     */
    public function ajax()
    {
        $aResponse = array();
        
        switch ($this->input->get('act', TYPE_STR))
        {
            case 'fav': # Добавление / удаление из избранных
            {
                $nItemID = $this->input->post('id', TYPE_UINT);
                $nTypeID = $this->input->post('type', TYPE_UINT);
                $nUserID = User::id();
                if( ! $nItemID || ! $nUserID || ! in_array($nTypeID, array(self::TYPE_VACANCY, self::TYPE_RESUME))) {
                    $this->ajaxResponse(Errors::IMPOSSIBLE);
                } 
                
                if( ! $this->security->validateToken()) {
                    $this->ajaxResponse(Errors::IMPOSSIBLE);
                }
                
                $bAdded = false;
                $sqlFilter = array('user_id'=>$nUserID,'item_id'=>$nItemID,'type_id'=>$nTypeID);
                $bExists = $this->db->select_data(TABLE_JOB_FAV, 'item_id', $sqlFilter, '', 1);
                if (empty($bExists)) {
                    $res = $this->db->insert(TABLE_JOB_FAV, $sqlFilter, false);
                    if (empty($res)) {
                        $this->ajaxResponse(Errors::IMPOSSIBLE);
                    }
                    $bAdded = true;
                } else {
                    $this->db->delete(TABLE_JOB_FAV, $sqlFilter);
                }
                
                $nCurrent = $this->security->userCounter('job_fav', ($bAdded ? 1 : -1));
                
                $this->ajaxResponse(array('res'=>$this->errors->no(), 'added'=>$bAdded, 'n'=>$nCurrent));
                
            } break;
            case 'company_contacts': # Контакты компании
            {
                $nCompanyID = $this->input->post('company_id', TYPE_UINT);
                if( ! $nCompanyID) break;
                $aData = $this->db->one_array('
                    SELECT I.id, I.agent_info, A.address as addr
                    FROM '.TABLE_ITEMS.' I, '.TABLE_ITEMS_ADDR.' A
                    WHERE I.id = '.$nCompanyID.' AND I.id = A.item_id AND I.agent_id = '.User::id().' AND I.moderated!=0');

                $emptyContacts = true;
                if (empty($aData)) {
                    $aData = array('id'=>$nCompanyID);
                } else {
                    if( ! empty($aData['agent_info'])) {
                        $aData['agent_info'] = unserialize($aData['agent_info']);
                        if( ! empty($aData['agent_info'])) {
                            $aData['phone'] = $aData['agent_info']['phone'];
                            $aData['email'] = $aData['agent_info']['email'];
                            $emptyContacts = false;
                        }
                    }
                    unset($aData['agent_info']);
                }
                if($emptyContacts) {
                    $aData = array_merge($aData, array('addr'=>'', 'phone'=>'', 'email'=>''));
                }
                $aData['city'] = Geo::defaultCity();
                $aResponse['contacts'] = $aData;
            } break;
            case 'item-claim': # Жалоба
            {
                $p = $this->input->postm(array(
                    'id'      => TYPE_UINT,
                    'type'    => TYPE_UINT,
                    'reasons' => TYPE_ARRAY_UINT,  
                    'comment' => TYPE_NOTAGS,
                    'captcha' => TYPE_STR,
                ));
                
                if( ! in_array($p['type'], array(self::TYPE_VACANCY, self::TYPE_RESUME))) $p['type'] = self::TYPE_VACANCY;
                
                $nUserID = User::id();
                
                do{
                    if( ! $p['id']) { $this->errors->impossible(); break; }
                    if (empty($p['reasons']) && $p['comment'] == '') { $this->errors->set( _t('job', 'Укажите причину и/или описание') ); break; }

                    if( ! $nUserID) {
                        $oProtection = new CCaptchaProtection();
                        if( !$oProtection->valid($this->input->cookie('c2'), $p['captcha']) ) {
                            $aResponse['captcha'] = 1;
                            $this->errors->set(_t('', 'Результат с картинки указан некорректно'), 'captcha');
                            break;
                        }
                    } else {
                        if( ! $this->security->validateToken()) {
                            $this->errors->reloadPage();
                            break;
                        }
                        # получаем дату последней добавленной пользователем жалобы
                        $sLastClaim = $this->db->one_data('SELECT created FROM '.TABLE_JOB_CLAIMS.' WHERE user_id = '.$nUserID.' ORDER BY created DESC LIMIT 1');
                        if( ! empty($sLastClaim)) {
                            $nMinutesTimeout = 3;
                            if( (time() - strtotime($sLastClaim)) < $nMinutesTimeout * 60 )
                            {
                                $this->errors->set( _t('job', 'Повторите попытку через 3 минуты') );
                                break;
                            }
                        }
                    }

                    $res = $this->db->insert(TABLE_JOB_CLAIMS, array(
                        'type_id' => $p['type'],
                        'item_id' => $p['id'],
                        'user_id' => $nUserID,
                        'comment' => $p['comment'],
                        'reasons' => array_sum($p['reasons']),
                        'ip'      => Request::remoteAddress(),
                        'created' => $this->db->now(),
                    ));
                    
                    if($res) {
                        if( ! $nUserID) Request::deleteCOOKIE('c2');
                        config::saveCount('job_claims', 1, true);
                    }
                                        
                } while(false);
          
            } break;
            case 'item-del': # Удаление вакансии / резюме
            {
                $nItemID = $this->input->post('id',   TYPE_UINT);
                $nTypeID = $this->input->post('type', TYPE_UINT);
                if( ! $nItemID || ! User::id() || !in_array($nTypeID, array(self::TYPE_VACANCY, self::TYPE_RESUME))) {
                    $this->ajaxResponse(Errors::IMPOSSIBLE);
                }
                
                if( ! $this->security->validateToken()) {
                    $this->errors->reloadPage();
                    $this->ajaxResponse( null );
                }

                $res = ( $nTypeID == self::TYPE_VACANCY ?
                            $this->vacancyDelete($nItemID) :
                            $this->resumeDelete($nItemID) );
                if($res) {
                    $this->security->userCounter( 'job_'.( $nTypeID == self::TYPE_VACANCY ? 'vacancy' : 'resume' ), -1 );
                    $this->ajaxResponse( Errors::SUCCESS );
                } else {
                    $this->ajaxResponse( Errors::IMPOSSIBLE );
                }
          
            } break;
            case 'item-promote': # Продвижение вакансии / резюме
            {
                do
                {
                    $nUserID = User::id();
                    $nItemID = $this->input->post('id', TYPE_UINT); // ID Объявления
                    $nSvcID = $this->input->post('svc', TYPE_UINT); // ID Услуги Job::SERVICE_
                    $sPaySystem = $this->input->post('ps', TYPE_NOTAGS); # Ключ способа оплаты
                    if (empty($sPaySystem)) $sPaySystem = Bills::PS_BALANCE;

                    if( ! $nItemID || ! $nSvcID || ! $nUserID || ! $this->security->validateToken() ) {
                        $this->errors->reloadPage();
                        break;
                    }

                    $aItem = $this->model->itemData($this->typeBySvc($nSvcID), $nItemID, array('id','user_id','status'));
                    if (empty($aItem) || $aItem['status'] == self::STATUS_NEW) {
                        $this->errors->reloadPage();
                        break;
                    }
                    if ($aItem['status'] == self::STATUS_BLOCKED) {
                        $this->errors->set( _t('', 'Невозможно продвинуть заблокированное объявление') );
                        break;
                    }
                    if ( ! User::isCurrent($aItem['user_id']) ) {
                        $this->errors->reloadPage();
                        break;
                    }

                    # проверяем способ оплаты
                    $aPaySystems = Bills::getPaySystems(true);
                    if (!isset($aPaySystems[$sPaySystem]) || empty($aPaySystems[$sPaySystem]['enabled'])) {
                        $this->errors->set( _t('bills', 'Выбранный способ оплаты на текущий момент недоступен') );
                        break;
                    }
                    # получаем стоимость услуги
                    $aSvc = Svc::model()->svcData($nSvcID);
                    if (empty($aSvc)) {
                        $this->errors->reloadPage();
                        break;
                    }
                    $nSvcPrice = $aSvc['price'];

                    $aPayAmount = Bills::getPayAmount($nSvcPrice, $sPaySystem);

                    $aResponse = $this->svc()->activateOrPay($this->module_name, $nSvcID, $nItemID,
                        $aPaySystems[$sPaySystem]['id'], $aPaySystems[$sPaySystem]['way'],
                        $nSvcPrice, $aPayAmount['amount'], $aPayAmount['currency']);
                    if ($aResponse['activated']) {
                        $aResponse['redirect'] = static::url('promote', array('success'=>1, 'id'=>$nItemID, 'svc'=>$nSvcID));
                    }

                } while(false);

                $this->ajaxResponseForm($aResponse);
            } break;
            case 'item-promote-data': # Данные о текущем состоянии активированных услуг вакансии / резюме
            {
                do
                {
                    $nUserID = User::id();
                    $nItemID = $this->input->post('item_id', TYPE_UINT);
                    $nTypeID = $this->input->post('type_id', TYPE_UINT);
                    if( ! in_array($nTypeID, array(self::TYPE_RESUME, self::TYPE_VACANCY))){
                        $nTypeID = self::TYPE_RESUME;
                    }

                    if( ! $nItemID || ! $nUserID ) {
                        $this->errors->reloadPage();
                        break;
                    }

                    $aItem = $this->model->itemData($nTypeID, $nItemID, array('id','user_id','status',
                        'svc', 'marked_to', 'fixed_to'));
                    if( empty($aItem)
                        || $aItem['status'] == self::STATUS_NEW
                        || $aItem['status'] == self::STATUS_BLOCKED
                        || ! User::isCurrent($aItem['user_id'])
                        || ! $this->security->validateReferer() ) {
                        $this->errors->reloadPage();
                        break;
                    }

                    switch ($nTypeID){
                        case Job::TYPE_RESUME:
                            $nServiceMark = Job::SERVICE_RESUME_MARK;
                            $nServiceFix = Job::SERVICE_RESUME_FIX;
                            break;
                        case Job::TYPE_VACANCY:
                            $nServiceMark = Job::SERVICE_VACANCY_MARK;
                            $nServiceFix = Job::SERVICE_VACANCY_FIX;
                            break;
                    }

                    $aResponse['marked'] = ($aItem['svc'] & $nServiceMark);
                    $aResponse['marked_to'] = _t('svc', 'Услуга уже активирована до <b>[date]</b>', array('date'=>tpl::date_format2($aItem['marked_to'], false)));
                    $aResponse['fixed'] = ($aItem['svc'] & $nServiceFix);
                    $aResponse['fixed_to'] = _t('svc', 'Услуга уже активирована до <b>[date]</b>', array('date'=>tpl::date_format2($aItem['fixed_to'], false)));
                } while(false);

                $this->ajaxResponseForm($aResponse);
            } break;

            default: $this->errors->impossible();
        }
        
        $aResponse['res'] = $this->errors->no();
        $this->ajaxResponse( $aResponse );
    }

    /**
     * Список избранных вакансий / резюме
     */
    public function fav()
    {
        $userID = User::id();
        bff::setActiveMenu('//job/fav');
        if ( ! $userID) {
            return $this->showForbidden( _t('', 'Избранные объявления'), _t('', 'Возможность добавлять объявления в избранные доступна только после авторизации.'), true);
        }

        # получаем список избранных вакансий
        $sql = array(
            'I.status = '.self::STATUS_PUBLICATED,
            'I.id = F.item_id AND F.type_id = '.self::TYPE_VACANCY.' AND F.user_id = '.$userID,
            'I.price_curr = CURR.id',
            'I.cat_id = C.id',
            'I.city_id = R.id',
        );
        if(static::premoderation()){
            $sql[] = 'I.moderated > 0';
        }


        $aData['vacancy'] = $this->db->select('SELECT I.id, I.title, I.link, I.created, I.company_id, I.company_title,
                                    I.price, (I.price * CURR.rate) as price_cross, I.price_curr, I.price_torg,
                                    CL.title as cat_title, M.link as item_link, ML.title as item_title, R.title_'.LNG.' as city_title
                                FROM '.TABLE_JOB_VACANCY.' I
                                     LEFT JOIN '.TABLE_ITEMS.' M ON M.id = I.company_id
                                     LEFT JOIN '.TABLE_ITEMS_LANG.' ML ON '.$this->db->langAnd(false, 'M', 'ML').',
                                     '.TABLE_JOB_FAV.' F,
                                     '.TABLE_CURRENCIES.' CURR,
                                     '.TABLE_JOB_CATEGORIES.' C,
                                     '.TABLE_JOB_CATEGORIES_LANG.' CL,
                                     '.TABLE_REGIONS.' R
                                WHERE '.join(' AND ', $sql).$this->db->langAnd(true, 'C', 'CL').'
                                ORDER BY I.created DESC');
        
        # получаем список избранных резюме
        $sql = array(
            'I.status = '.self::STATUS_PUBLICATED,
            'I.id = F.item_id AND F.type_id = '.self::TYPE_RESUME.' AND F.user_id = '.$userID,
            'I.price_curr = CURR.id',
            'I.cat_id = C.id',
            'I.city_id = R.id',
        );
        if(static::premoderation()){
            $sql[] = 'I.moderated > 0';
        }
        $aData['resume'] = $this->db->select('SELECT I.id, I.title, I.link, I.created, I.fio, '.self::resumeYears.',
                                    I.price, (I.price * CURR.rate) as price_cross, I.price_curr, I.price_torg,
                                    C.title as cat_title, R.title_'.LNG.' as city_title
                                FROM '.TABLE_JOB_RESUME.' I,
                                     '.TABLE_JOB_FAV.' F,
                                     '.TABLE_CURRENCIES.' CURR,
                                     '.TABLE_JOB_CATEGORIES.' C,
                                     '.TABLE_JOB_CATEGORIES_LANG.' CL,
                                     '.TABLE_REGIONS.' R
                                WHERE '.join(' AND ', $sql).$this->db->langAnd(true, 'C', 'CL').'
                                ORDER BY I.created DESC');

        $aData['tab'] = ( empty($aData['resume']) && ! empty($aData['vacancy']) ? self::TYPE_VACANCY : self::TYPE_RESUME );
        $this->security->userCounterCheck('job_fav', (sizeof($aData['vacancy']) + sizeof($aData['resume'])) );
        $this->initRightblock(__FUNCTION__);
        bff::setMeta(_t('job', 'Избранные вакансии и резюме'));
        return $this->viewPHP($aData, 'fav');
    }
    
    # ---------------------------------------------------------------------------------
    # Вакансии

    /**
     * Вакансии: главная
     */
    public function vacancy_index()
    {
        $aData = $this->search_filter(self::TYPE_VACANCY);

        if( $this->regionsFilterEnabled() ) {
            $aData['cats'] = $this->db->select('SELECT C.id, C.keyword, CL.title as t, COUNT(I.id) as i
                FROM '.TABLE_JOB_CATEGORIES.' C
                    LEFT JOIN '.TABLE_JOB_VACANCY.' I ON I.cat_id = C.id AND I.status = '.self::STATUS_PUBLICATED.'
                        '.(static::premoderation() ? ' AND I.moderated > 0 ' : '' ).'
                        AND I.city_id = '.Geo::cityID().'
                    , '.TABLE_JOB_CATEGORIES_LANG.' CL
                WHERE C.enabled = 1 '.$this->db->langAnd(true, 'C', 'CL').'
                GROUP BY C.id
                ORDER BY C.num');
        } else {
            $aData['cats'] = $this->db->select('
                SELECT C.id, C.keyword, CL.title as t, C.vacancy_items as i
                FROM '.TABLE_JOB_CATEGORIES.' C, '.TABLE_JOB_CATEGORIES_LANG.' CL
                WHERE C.enabled = 1 '.$this->db->langAnd(true, 'C', 'CL').'
                ORDER BY C.num');
        }

        # SEO: Банк вакансий (главная)
        $this->urlCorrection(static::url('vacancy.list'));
        $this->seo()->canonicalUrl(static::url('vacancy.list', array(), true));
        $this->setMeta('index-'.self::TYPE_VACANCY, array('region' => Geo::filter('title')), $aData);

        $aData['latest'] = $this->vacancy_minilist(6);
        $this->initRightblock(__FUNCTION__, array('filter'=>&$aData['filter']));
        return $this->viewPHP($aData, 'vacancy.index');
    }

    /**
     * Вакансии: добавление
     */
    public function vacancy_add()
    {
        if( ! empty($_GET['success'])) {
            bff::setActiveMenu('//job/vacancy', true);
            if(static::premoderation()){
                return $this->showSuccess(_t('job', 'Новая вакансия'), _t('job', 'Вы успешно разместили вакансию. После проверки (1-2 дня), мы разместим ее на сайте.'));
            }else{
                return $this->showSuccess(_t('job', 'Новая вакансия'), _t('job', 'Вы успешно разместили вакансию.'));
            }
        }
        
        $nUserID = User::id();
        
        if( ! $nUserID) {
            bff::setActiveMenu('//job/vacancy', true);
            return $this->showForbidden(_t('job', 'Новая вакансия'), _t('job', 'Возможность размещать вакансии доступна только авторизованным пользователям.'), true);
        }
        
        $aData = $this->processVacancyData(true, $aFields);

        $aAgentCompany = Users::model()->agentCompanies($nUserID);
        
        if (Request::isPOST())
        {
            if($aData['company_id']>0) {
                $bFound = false;
                foreach($aAgentCompany as $v) {
                    if($v['id'] == $aData['company_id']) {
                        $bFound = true;
                        break;
                    }
                }
                if( ! $bFound) {
                    $this->errors->set( _t('job', 'Укажите компанию из списка доступных') );
                }
            }
            
            if( $this->errors->no() ) {
                if( ! $this->security->validateToken() ) {
                    $this->errors->reloadPage();
                }
            }
            
            if($this->errors->no()) 
            {
                # создаем вакансию
                $aData['created'] = $aData['publicated'] = $aData['modified'] = $this->db->now();
                $aData['publicated_to'] = $this->getPublicatedTo('vacancy');
                $aData['publicated_order'] = $this->db->now();
                $aData['status'] = static::premoderation() ? self::STATUS_NEW : self::STATUS_PUBLICATED;
                $aData['moderated'] = 0;
                $aData['user_id'] = $nUserID;
                
                $nItemID = $this->db->insert(TABLE_JOB_VACANCY, $aData);
                if( ! $nItemID) {
                    $this->errors->set( _t('job', 'Неудалось разместить вакансию, обратитесь к администрации') );
                } else {
                    $this->db->update(TABLE_JOB_VACANCY, array(
                        'link' => static::urlViewVacancy($nItemID, $aData['keyword'], $aData['city_id']),
                    ), array('id'=>$nItemID));
                    $this->security->userCounter('job_vacancy', 1);
                    if ( ! static::premoderation()) {
                        $this->itemsCounterUpdate($aData['cat_id'], 'vacancy', true);
                    } else {
                        $this->moderationCounter(self::TYPE_VACANCY);
                    }
                }
            }
            
            $aResponse = array('res'=>$this->errors->no(), 'fields'=>$aFields);
            if($aResponse['res']) {
                $aResponse['redirect'] = Job::url('vacancy.add', array('success'=>1));
            }
            $this->ajaxResponse( $aResponse );
        }

        $aData['cats'] = $this->categoriesOptions($aData['cat_id'], _t('job','Выбрать'));
        $aData['schedule_html'] = $this->getSchedule($aData['schedule'], 'radio', 'schedule');
        $aData['curr'] = Site::currencyOptions($aData['price_curr'], false);
        $aData['token'] = $this->security->getToken();
        $aData['agent_company'] = &$aAgentCompany;
        $aData['id'] = 0;

        # SEO: Добавление вакансии
        $this->urlCorrection(static::url('vacancy.add'));
        $this->seo()->canonicalUrl(static::url('vacancy.add', array(), true));
        $this->setMeta('add-'.self::TYPE_VACANCY, array('region' => Geo::filter('title')), $aData);

        bff::showInstruction('job_vacancy_add');
        bff::setActiveMenu('//job/vacancy', true);
        $this->initRightblock(__FUNCTION__);
        return $this->viewPHP($aData, 'vacancy.form');
    }

    /**
     * Вакансии: редактирование
     */
    public function vacancy_edit()
    {
        $nItemID = $this->input->getpost('id', TYPE_UINT);
        if( ! $nItemID) $this->errors->error404();

        $nUserID = User::id();
        
        if( ! $nUserID) {
            bff::setActiveMenu('//job/vacancy', true);
            return $this->showForbidden(_t('job', 'Редактирование вакансии'), _t('job', 'Возможность редактирования вакансий доступна только авторизованным пользователям.') , true);
        }
        
        $aData = $this->db->one_array('SELECT * FROM '.TABLE_JOB_VACANCY.' WHERE id = :itemId AND user_id = :userId', array(':itemId'=>$nItemID, ':userId'=>$nUserID));
        if (empty($aData)) {
            $this->errors->error404();
        }
        
        $nStatus = $aData['status'];

        $aAgentCompany = Users::model()->agentCompanies($nUserID);

        if (Request::isPOST())
        {
            $aDataOld = $aData;
            $aData = $this->processVacancyData(false, $aFields);
            
            if($aData['company_id']>0) {
                $bFound = false;
                foreach($aAgentCompany as $v) {
                    if($v['id'] == $aData['company_id']) {
                        $bFound = true;
                        break;
                    }
                }
                
                if( ! $bFound) {
                    $this->errors->set( _t('job', 'Укажите компанию из списка доступных') );
                }
            }

            if( $this->errors->no() ) {
                if( ! $this->security->validateToken() ) {
                    $this->errors->reloadPage();
                }
            }
            if($this->errors->no()) 
            {
                // редактируем вакансию
                $aData['modified'] = $this->db->now();
                if($nStatus == self::STATUS_BLOCKED) {
                    $aData['moderated'] = 0; // отправляем на повторную проверку модератору
                } else {
                    if((int)$aDataOld['moderated'] == 0) {
                        $aData['moderated'] = 2;
                    }
                }
                $aData['link'] = static::urlViewVacancy($nItemID, $aData['keyword'], $aData['city_id']);
                
                $res = $this->db->update(TABLE_JOB_VACANCY, $aData, array('id'=>$nItemID));
                if (empty($res)) {
                    $this->errors->set( _t('job', 'Неудалось отредактировать вакансию, обратитесь к администрации') );
                }
            }
            
            $aResponse = array('res'=>$this->errors->no(), 'fields'=>$aFields);
            if($aResponse['res']) {
                $aResponse['redirect'] = Job::url('view', $aData['link']);
            }
            $this->ajaxResponse( $aResponse );
        } else {
            
        }

        $aData['cats'] = $this->categoriesOptions($aData['cat_id'], _t('job','Выбрать'));
        $aData['schedule_html'] = $this->getSchedule($aData['schedule'], 'radio', 'schedule');
        $aData['curr']  = Site::currencyOptions($aData['price_curr'], false);
        $aData['token'] = $this->security->getToken();
        $aData['agent_company'] = &$aAgentCompany;
        
        bff::showInstruction('job_vacancy_edit');
        bff::setActiveMenu('//job/vacancy', true);
        bff::setMeta(_t('job', 'Редактирование вакансии'));
        $this->initRightblock(__FUNCTION__);
        return $this->viewPHP($aData, 'vacancy.form');
    }

    /**
     * Вакансии: просмотр
     */
    public function vacancy_view()
    {
        bff::setActiveMenu('//job/vacancy', true);

        $nItemID = $this->input->getpost('id', TYPE_UINT);
        if( ! $nItemID) $this->errors->error404();

        $nUserID = User::id();
        $isPrint  = $this->input->get('print', TYPE_BOOL);
        
        $aData = $this->db->one_array('SELECT I.*, '.($nUserID ? ' (F.item_id)' : '0').' as fav, CL.title as cat_title, R.title_'.LNG.' as city_title, C.keyword AS cat_keyword
                                FROM '.TABLE_JOB_VACANCY.' I
                                     '.($nUserID>0 ? ' LEFT JOIN '.TABLE_JOB_FAV.' F ON F.item_id = I.id AND F.user_id = '.$nUserID.' AND F.type_id = '.self::TYPE_VACANCY : '').',
                                     '.TABLE_JOB_CATEGORIES.' C,
                                     '.TABLE_JOB_CATEGORIES_LANG.' CL,
                                     '.TABLE_REGIONS.' R
                                WHERE I.id = '.$nItemID.' 
                                  AND I.cat_id = C.id
                                  AND I.city_id = R.id '.
                                  $this->db->langAnd(true, 'C', 'CL'));
        if (empty($aData)) {
            $this->errors->error404();
        }

        if($aData['status'] != self::STATUS_PUBLICATED)
        {
            $titleForbidden = _t('job', 'Вакансия #[id]', array('id'=>$nItemID));

            if(static::premoderation()) {
                if ($aData['status'] == self::STATUS_NEW || ! $aData['moderated']) {
                    return $this->showForbidden($titleForbidden, _t('job', 'Вакансия ожидает проверки модератора'));
                }
            }
            if($aData['status'] == self::STATUS_BLOCKED) {
                return $this->showForbidden($titleForbidden, _t('job', 'Вакансия заблокирована по причине:<br/><b>[reason]</b>', array('reason'=>nl2br($aData['blocked_reason']))));
            }
            else if($aData['status'] == self::STATUS_PUBLICATED_OUT) {
                return $this->showForbidden($titleForbidden, _t('job', 'Срок публикации вакансии истек'));
            }
        }
        
        if ($aData['company_id'] > 0) {
            $aData['item'] = $this->db->one_array('
                SELECT I.id, I.link, IL.title, I.imgcnt, I.img_list
                FROM '.TABLE_ITEMS.' I,
                     '.TABLE_ITEMS_LANG.' IL
                WHERE I.id = '.$aData['company_id'].$this->db->langAnd(true, 'I','IL'));
        } else {
            if ($aData['company_cityid'] > 0) {
                $aData['company_city_title'] = Geo::regionTitle($aData['company_cityid']);
            }
        }

        # SEO: Просмотр вакансии
        $aData['url'] = static::url('view', $aData['link']);
        $this->urlCorrection($aData['url']);
        $this->seo()->canonicalUrl(static::url('view', $aData['link'], true));
        $this->setMeta('view-'.self::TYPE_VACANCY, array(
            'title'       => $aData['title'],
            'description' => tpl::truncate($aData['description'], 150),
            'region'      => Geo::filter('title'),
            'company'     => ( $aData['company_id'] > 0 ? $aData['item']['title'] :
                                ( ! empty($aData['company_title']) ? $aData['company_title'] : '') ),
        ), $aData);
        if ( ! $isPrint) {
            $this->seo()->setSocialMetaOG($aData['share_title'], $aData['share_description'], array(), $aData['url'], $aData['share_sitename']);
        }
        
        if($isPrint) {
            View::setLayout('print');
            return $this->viewPHP($aData, 'vacancy.view.print');
        }

        if( ! $nUserID || ($aData['user_id'] > 0 && $aData['user_id']!=$nUserID)) {
            $this->db->exec('UPDATE '.TABLE_JOB_VACANCY.' SET views = views + 1 WHERE id = '.$nItemID);
        }
        
        $aData['similar'] = $this->vacancy_minilist(6, 
                    array('I.id != '.$nItemID, 
                          'I.cat_id = '.$aData['cat_id'], 
                          'I.city_id = '.$aData['city_id']));

        $this->initRightblock(__FUNCTION__);
        return $this->viewPHP($aData, 'vacancy.view');
    }

    /**
     * Вакансии: список в профиле
     */
    public function vacancy_profile()
    {
        $sql = array(
            'I.user_id = '.User::id(),
            'I.status != '.self::STATUS_PUBLICATED_OUT,
            'I.price_curr = CURR.id',
            'I.cat_id = C.id', );
            
        $aData['items'] = $this->db->select('SELECT I.id, I.title, I.link, I.created, I.status, I.moderated, I.publicated_to, I.company_id, I.company_title,
                                    I.price, (I.price * CURR.rate) as price_cross, I.price_curr, I.price_torg,
                                    I.svc, (I.svc & '.Job::SERVICE_VACANCY_FIX.') as fixed, (I.svc & '.Job::SERVICE_VACANCY_MARK.') as marked,
                                    CL.title as cat_title, R.title_'.LNG.' as city_title,
                                    ML.title as item_title, M.link as item_link
                                FROM '.TABLE_JOB_VACANCY.' I
                                    LEFT JOIN '.TABLE_REGIONS.' R ON I.city_id = R.id
                                    LEFT JOIN '.TABLE_ITEMS.' M ON M.id = I.company_id
                                    LEFT JOIN '.TABLE_ITEMS_LANG.' ML ON '.$this->db->langAnd(false, 'M', 'ML').',
                                     '.TABLE_CURRENCIES.' CURR,
                                     '.TABLE_JOB_CATEGORIES.' C,
                                     '.TABLE_JOB_CATEGORIES_LANG.' CL
                                WHERE '.join(' AND ', $sql).$this->db->langAnd(true, 'C', 'CL').'
                                ORDER BY I.created DESC');
        
        $this->security->userCounterCheck( 'job_vacancy', sizeof($aData['items']) );
        $this->initRightblock(__FUNCTION__);
        bff::setMeta(_t('job', 'Мои вакансии - Кабинет'));
        return $this->viewPHP($aData, 'vacancy.profile');
    }

    /**
     * Вакансии: список
     */
    protected function vacancy_minilist($nLimit, $sql = array())
    {
        $sql[] = 'I.status = '.self::STATUS_PUBLICATED;
        if(static::premoderation()){
            $sql[] = 'I.moderated > 0';
        }
        $sql[] = 'I.price_curr = CURR.id';
        $sql[] = 'I.cat_id = C.id';
        $sql[] = 'I.city_id = R.id';
        if( $this->regionsFilterEnabled() ) $sql[] = 'I.city_id = '.Geo::cityID();
        
        $nUserID = User::id();

        return $this->db->select('SELECT I.id, I.title, I.link, I.created, I.company_id, I.company_title, I.price, (I.price * CURR.rate) as price_cross, I.price_curr, I.price_torg,
                                    CL.title as cat_title, M.link as item_link, ML.title as item_title, R.title_'.LNG.' as city_title,
                                    I.svc, (I.svc & '.self::SERVICE_VACANCY_FIX.') as fixed, (I.svc & '.self::SERVICE_VACANCY_MARK.') as marked
                                    '.($nUserID ? ', (F.item_id)' : ',0').' as fav
                                FROM '.TABLE_JOB_VACANCY.' I
                                     LEFT JOIN '.TABLE_ITEMS.' M ON M.id = I.company_id
                                     LEFT JOIN '.TABLE_ITEMS_LANG.' ML ON '.$this->db->langAnd(false, 'M', 'ML').'
                                     '.($nUserID ? ' LEFT JOIN '.TABLE_JOB_FAV.' F ON I.id = F.item_id AND F.user_id = '.$nUserID.' AND F.type_id = '.self::TYPE_VACANCY : '').'
                                     ,
                                     '.TABLE_CURRENCIES.' CURR,
                                     '.TABLE_JOB_CATEGORIES.' C,
                                     '.TABLE_JOB_CATEGORIES_LANG.' CL,
                                     '.TABLE_REGIONS.' R
                                WHERE '.join(' AND ', $sql).$this->db->langAnd(true, 'C', 'CL').'
                                ORDER BY I.created DESC
                                '.( $nLimit!==false ? $this->db->prepareLimit(0,$nLimit) : '' ) );
    }

    /**
     * Вакансии: список вакансий компании
     */
    public function vacancy_company($nCompanyID = 0, array $aData = array())
    {
        $nTotal = $this->model->companyVacancy($nCompanyID, true);
        $aData['pgn'] = $this->generatePagenationDots($nTotal, 10, 3, Items::url('view', array('link'=>$aData['link'], 'tab'=>'vacancy', 'q'=>'page={page}')), $sqlLimit);

        $aData['items'] = $this->model->companyVacancy($nCompanyID, false, $sqlLimit);
        return $this->viewPHP($aData, 'vacancy.company');
    }
    
    # ---------------------------------------------------------------------------------
    # Резюме

    /**
     * Резюме: главная
     */
    public function resume_index()
    {
        $aData = $this->search_filter(self::TYPE_RESUME);

        if( $this->regionsFilterEnabled() ) {
            $aData['cats'] = $this->db->select('SELECT C.id, C.keyword, CL.title as t, COUNT(I.id) as i
                FROM '.TABLE_JOB_CATEGORIES.' C
                    LEFT JOIN '.TABLE_JOB_RESUME.' I ON I.cat_id = C.id AND I.status = '.self::STATUS_PUBLICATED.'
                        '.(static::premoderation() ? ' AND I.moderated > 0 ' : '' ).'
                        AND I.city_id = '.Geo::cityID().'
                    , '.TABLE_JOB_CATEGORIES_LANG.' CL
                WHERE C.enabled = 1 '.$this->db->langAnd(true, 'C', 'CL').'
                GROUP BY C.id
                ORDER BY C.num');
        } else {
            $aData['cats'] = $this->db->select('
                SELECT C.id, C.keyword, CL.title as t, C.resume_items as i
                FROM '.TABLE_JOB_CATEGORIES.' C,
                     '.TABLE_JOB_CATEGORIES_LANG.' CL
                WHERE C.enabled = 1 '.$this->db->langAnd(true, 'C', 'CL').'
                ORDER BY C.num');
        }

        # SEO: Банк резюме (главная)
        $this->urlCorrection(static::url('resume.list'));
        $this->seo()->canonicalUrl(static::url('resume.list', array(), true));
        $this->setMeta('index-'.self::TYPE_RESUME, array('region' => Geo::filter('title')), $aData);

        $aData['latest'] = $this->resume_minilist(6);
        $this->initRightblock(__FUNCTION__, array('filter'=>&$aData['filter']));
        return $this->viewPHP($aData, 'resume.index');
    }

    /**
     * Резюме: добавление
     */
    public function resume_add()
    {
        if( ! empty($_GET['success'])) {
            bff::setActiveMenu('//job/resume', true);
            bff::setMeta(_t('job', 'Новое резюме'));
            return $this->showSuccess(_t('job', 'Новое резюме'), _t('job', 'Вы успешно разместили резюме. После проверки (1-2 дня), мы разместим ее на сайте.'));
        }

        $nUserID = User::id();
        
        if( ! $nUserID) {
            bff::setActiveMenu('//job/resume', true);
            bff::setMeta(_t('job', 'Новое резюме'));
            return $this->showForbidden(_t('job', 'Новое резюме'), _t('job', 'Возможность размещать резюме доступна только авторизованным пользователям.'), true);
        }
        
        $aData = $this->processResumeData(true, $aFields);

        if (Request::isPOST())
        {
            if( $this->errors->no() ) {
                if( ! $this->security->validateToken() ) {
                    $this->errors->reloadPage();
                }
            }
            if($this->errors->no()) 
            {
                // создаем резюме
                $aData['created'] = $aData['publicated'] = $aData['modified'] = $this->db->now();
                $aData['publicated_to'] = $this->getPublicatedTo('resume');
                $aData['publicated_order'] = $this->db->now();
                $aData['status'] = self::STATUS_PUBLICATED;
                $aData['moderated'] = 0;
                $aData['user_id'] = $nUserID;
                
                $nItemID = $this->db->insert(TABLE_JOB_RESUME, $aData);
                if( ! $nItemID) {
                    $this->errors->set( _t('job', 'Неудалось разместить резюме, обратитесь к администрации') );
                } else {
                    $this->db->update(TABLE_JOB_RESUME, array(
                        'link' => static::urlViewResume($nItemID, $aData['keyword'], $aData['city_id']),
                    ), array('id'=>$nItemID));
                    $this->security->userCounter('job_resume', 1);
                    if ( ! static::premoderation()) {
                         $this->itemsCounterUpdate($aData['cat_id'], 'resume', true);
                    } else {
                        $this->moderationCounter(self::TYPE_RESUME);
                    }
                }
            }
            
            $aResponse = array('res'=>$this->errors->no(), 'fields'=>$aFields);
            if($aResponse['res']) {
                $aResponse['redirect'] = Job::url('resume.add', array('success'=>1, 'id'=>$nItemID));
            }
            $this->ajaxResponse( $aResponse );
        }

        $aData['experience_html'] = $this->getExperience($aData['experience'], 'options');
        $aData['cats'] = $this->categoriesOptions($aData['cat_id'], _t('job','выберите из списка'));
        $aData['schedule_html'] = $this->getSchedule($aData['schedule'], 'radio', 'schedule');
        $aData['curr'] = Site::currencyOptions($aData['price_curr'], false);
        $aData['token'] = $this->security->getToken();
        $aData['phones'] = $this->resumePhonesPrepareEdit( $aData['phones'] );
        $aData['id'] = 0;

        # SEO: Добавление резюме
        $this->urlCorrection(static::url('resume.add'));
        $this->seo()->canonicalUrl(static::url('resume.add', array(), true));
        $this->setMeta('add-'.self::TYPE_RESUME, array('region' => Geo::filter('title')), $aData);

        bff::showInstruction('job_resume_add');
        bff::setActiveMenu('//job/resume', true);
        $this->initRightblock(__FUNCTION__);
        return $this->viewPHP($aData, 'resume.form');
    }

    /**
     * Резюме: редактирование
     */
    public function resume_edit()
    {
        $nItemID = $this->input->getpost('id', TYPE_UINT);
        if( ! $nItemID) $this->errors->error404();

        $nUserID = User::id();
        
        if( ! $nUserID) {
            bff::setActiveMenu('//job/resume', true);
            return $this->showForbidden(_t('job', 'Редактирование резюме'), _t('job', 'Возможность редактирования резюме доступна только авторизованным пользователям.'), true);
        }
        
        $aData = $this->db->one_array('SELECT * FROM '.TABLE_JOB_RESUME.' WHERE id = :id AND user_id = :userId', array(':id'=>$nItemID, ':userId'=>$nUserID));
        if (empty($aData)) {
            $this->errors->error404();
        }
        
        $nStatus = $aData['status'];

        if (Request::isPOST())
        {
            $aOldData = $aData;
            $aData = $this->processResumeData(false, $aFields);

            if( $this->errors->no() ) {
                if( ! $this->security->validateToken() ) {
                    $this->errors->reloadPage();
                }
            }
            if($this->errors->no()) 
            {
                // редактируем резюме
                $aData['modified'] = $this->db->now();
                if($nStatus == self::STATUS_BLOCKED) {
                    $aData['moderated'] = 0; // отправляем на повторную проверку модератору
                } else {
                    if($aOldData['moderated']){
                        $aData['moderated'] = 2;
                    }
                }
                $aData['link'] = static::urlViewResume($nItemID, $aData['keyword'], $aData['city_id']);
                
                $res = $this->db->update(TABLE_JOB_RESUME, $aData, array('id'=>$nItemID));
                if (empty($res)) {
                    $this->errors->set( _t('job', 'Неудалось отредактировать резюме, обратитесь к администрации') );
                }
            }
            
            $aResponse = array('res'=>$this->errors->no(), 'fields'=>$aFields);
            if($aResponse['res']) {
                $aResponse['redirect'] = Job::url('view', $aData['link']);
            }
            $this->ajaxResponse( $aResponse );
        } else {
            
        }

        $aData['experience_html'] = $this->getExperience($aData['experience'], 'options');
        $aData['cats'] = $this->categoriesOptions($aData['cat_id'], false);
        $aData['schedule_html'] = $this->getSchedule($aData['schedule'], 'radio', 'schedule');
        $aData['curr'] = Site::currencyOptions($aData['price_curr'], false);
        $aData['token'] = $this->security->getToken();
        $aData['phones'] = $this->resumePhonesPrepareEdit( $aData['phones'] );

        bff::showInstruction('job_resume_edit');
        bff::setActiveMenu('//job/resume', true);
        bff::setMeta(_t('job', 'Редактирование резюме'));
        $this->initRightblock(__FUNCTION__);
        return $this->viewPHP($aData, 'resume.form');
    }

    /**
     * Резюме: просмотр
     */
    public function resume_view()
    {
        bff::setActiveMenu('//job/resume', true);

        $nItemID = $this->input->getpost('id', TYPE_UINT);
        if ( ! $nItemID) $this->errors->error404();

        $nUserID = User::id();
        $isPrint  = $this->input->get('print', TYPE_BOOL);
        
        $aData = $this->db->one_array('SELECT I.*, '.self::resumeYears.', '.($nUserID ? ' (F.item_id)' : '0').' as fav,
                C.keyword as cat_keyword, CL.title as cat_title, R.title_'.LNG.' as city_title
            FROM '.TABLE_JOB_RESUME.' I
                 '.($nUserID>0 ? ' LEFT JOIN '.TABLE_JOB_FAV.' F ON F.item_id = I.id AND F.user_id = '.$nUserID.' AND F.type_id = '.self::TYPE_RESUME : '').',
                 '.TABLE_JOB_CATEGORIES.' C,
                 '.TABLE_JOB_CATEGORIES_LANG.' CL,
                 '.TABLE_REGIONS.' R
            WHERE I.id = :id
              AND I.cat_id = C.id
              AND I.city_id = R.id '.
              $this->db->langAnd(true, 'C', 'CL'), array(':id'=>$nItemID));
        if (empty($aData)) {
            $this->errors->error404();
        }

        if($aData['status'] != self::STATUS_PUBLICATED)
        {
            $titleForbidden = _t('job', 'Резюме #[id]', array('id'=>$nItemID));
            if(static::premoderation()){
                if ($aData['status'] == self::STATUS_NEW || ! $aData['moderated']) {
                    return $this->showForbidden($titleForbidden, _t('job', 'Резюме ожидает проверки модератора'));
                }
            }
            if ($aData['status'] == self::STATUS_BLOCKED) {
                return $this->showForbidden($titleForbidden, _t('job', 'Резюме заблокировано по причине:<br/><b>[reason]</b>', array('reason'=>nl2br($aData['blocked_reason']))));
            } else if ($aData['status'] == self::STATUS_PUBLICATED_OUT) {
                return $this->showForbidden($titleForbidden, _t('job', 'Срок публикации резюме истек'));
            }
        }
        
        $aData['phones'] = $this->resumePhonesPrepareEdit($aData['phones']);

        # SEO: Просмотр резюме
        $aData['url'] = static::url('view', $aData['link']);
        $this->urlCorrection($aData['url']);
        $this->seo()->canonicalUrl(static::url('view', $aData['link'], true));
        $this->setMeta('view-'.self::TYPE_RESUME, array(
            'title'       => $aData['title'],
            'description' => tpl::truncate($aData['description'], 150),
            'region'      => Geo::filter('title'),
        ), $aData);
        if ( ! $isPrint) {
            $this->seo()->setSocialMetaOG($aData['share_title'], $aData['share_description'], array(), $aData['url'], $aData['share_sitename']);
        }

        if ($isPrint) {
            View::setLayout('print');
            return $this->viewPHP($aData, 'resume.view.print');
        }

        if ( ! $nUserID || ($aData['user_id'] > 0 && $aData['user_id']!=$nUserID)) {
            $this->db->update(TABLE_JOB_RESUME, array('views = views + 1'), array('id'=>$nItemID));
        }
        
        $aData['similar'] = $this->resume_minilist(6, 
                                array('I.id != '.$nItemID,
                                      'I.cat_id = '.$aData['cat_id'],
                                      'I.city_id = '.$aData['city_id']));

        $this->initRightblock(__FUNCTION__);
        return $this->viewPHP($aData, 'resume.view');
    }

    /**
     * Резюме: список в профиле пользователя
     */
    public function resume_profile()
    {
        $sql = array(
            'I.user_id = '.User::id(),
            'I.status != '.self::STATUS_PUBLICATED_OUT,
            'I.price_curr = CURR.id',
            'I.cat_id = C.id', );
            
        $aData['items'] = $this->db->select('SELECT I.id, I.title, I.link, I.created, I.status, I.moderated, I.publicated_to, I.fio, '.self::resumeYears.',
                                    I.price, (I.price * CURR.rate) as price_cross, I.price_curr, I.price_torg,
                                    I.svc, (I.svc & '.Job::SERVICE_RESUME_FIX.') as fixed, (I.svc & '.Job::SERVICE_RESUME_MARK.') as marked,
                                    CL.title as cat_title, R.title_'.LNG.' as city_title
                                FROM '.TABLE_JOB_RESUME.' I
                                    LEFT JOIN '.TABLE_REGIONS.' R ON I.city_id = R.id,
                                     '.TABLE_CURRENCIES.' CURR,
                                     '.TABLE_JOB_CATEGORIES.' C,
                                     '.TABLE_JOB_CATEGORIES_LANG.' CL
                                WHERE '.join(' AND ', $sql).$this->db->langAnd(true, 'C', 'CL').'
                                ORDER BY I.created DESC');
        
        $this->security->userCounterCheck( 'job_resume', sizeof($aData['items']) );
        $this->initRightblock(__FUNCTION__);
        bff::setMeta(_t('job', 'Мои резюме - Кабинет'));
        return $this->viewPHP($aData, 'resume.profile');
    }

    /**
     * Резюме: список
     */
    protected function resume_minilist($nLimit, $sql = array())
    {
        $sql[] = 'I.status = '.self::STATUS_PUBLICATED;
        if(static::premoderation()){
            $sql[] = 'I.moderated > 0';
        }
        $sql[] = 'I.price_curr = CURR.id';
        $sql[] = 'I.cat_id = C.id';
        $sql[] = 'I.city_id = R.id';
        if( $this->regionsFilterEnabled() ) $sql[] = 'I.city_id = '.Geo::cityID();
        
        $nUserID = User::id();
        
        return $this->db->select('SELECT I.id, I.title, I.link, I.created, I.fio, '.self::resumeYears.', '.($nUserID ? ' (F.item_id)' : '0').' as fav, I.price, (I.price * CURR.rate) as price_cross, I.price_curr, I.price_torg,
                                    CL.title as cat_title, R.title_'.LNG.' as city_title,
                                    I.svc, (I.svc & '.self::SERVICE_RESUME_FIX.') as fixed, (I.svc & '.self::SERVICE_RESUME_MARK.') as marked
                                FROM '.TABLE_JOB_RESUME.' I
                                     '.($nUserID ? ' LEFT JOIN '.TABLE_JOB_FAV.' F ON I.id = F.item_id AND F.user_id = '.$nUserID.' AND F.type_id = '.self::TYPE_RESUME : '').',
                                     '.TABLE_CURRENCIES.' CURR,
                                     '.TABLE_JOB_CATEGORIES.' C,
                                     '.TABLE_JOB_CATEGORIES_LANG.' CL,
                                     '.TABLE_REGIONS.' R
                                WHERE '.join(' AND ', $sql).$this->db->langAnd(true, 'C', 'CL').'
                                ORDER BY I.created DESC
                                '.$this->db->prepareLimit(0,$nLimit));
    }

    /**
     * Компании: список
     */
    public function company()
    {
        $sql = 'I.vacancy > 0 AND I.enabled = 1 AND I.id = IA.item_id';
        if( $this->regionsFilterEnabled() ) {
            $sql .= ' AND IA.city_id = '.Geo::cityID();
        }
        if(static::premoderation()){
            $sql .= ' AND I.moderated > 0';
        }


        $nPerpage = 10;
        $nTotal = $this->db->one_data('SELECT COUNT(I.id) FROM '.TABLE_ITEMS.' I, '.TABLE_ITEMS_ADDR.' IA WHERE '.$sql);
        $aData['pgn'] = $this->generatePagenationDots($nTotal, $nPerpage, 2, static::url('company.list', 'page={page}'), $sqlLimit);
        $nPage = $this->input->get('page', TYPE_UINT); if (!$nPage) $nPage = 1;

        $aData['items'] = $this->db->select('
                SELECT I.id, IL.title, I.vacancy, IL.descshort as description, I.img_list as img, I.imgcnt, I.link
                FROM '.TABLE_ITEMS.' I, '.TABLE_ITEMS_LANG.' IL, '.TABLE_ITEMS_ADDR.' IA
                WHERE '.$sql.$this->db->langAnd(true, 'I','IL').'
                GROUP BY I.id '.$sqlLimit);
        
        if( ! empty($aData['items']))
        {
            foreach($aData['items'] as &$v) {
                $v['link_vacancy'] = Items::url('view', array('link'=>$v['link'], 'tab'=>'vacancy'));
                $v['link'] = Items::url('view', $v['link']);
            } unset($v);
        }

        # SEO: Компании - список
        $this->urlCorrection(static::url('company.list'));
        $this->seo()->canonicalUrl(static::url('company.list', array(), true), array(
            'page' => $nPage
        ));
        $this->setMeta('company', array(
            'region' => Geo::filter('title'),
            'page'   => $nPage
        ), $aData);

        bff::setActiveMenu('//job/company');
        $this->initRightblock(__FUNCTION__);
        return $this->viewPHP($aData, 'company.listing');
    }

    /**
     * Агентства: список
     */
    public function agents()
    {
        $categoryID = static::agentsCategory();
        $sql = 'I.id = C.item_id AND C.category_id = '.$categoryID.' AND I.enabled = 1 AND I.moderated = 1 AND I.id = IA.item_id';
        if ($this->regionsFilterEnabled()) {
            $sql .= ' AND IA.city_id = '.Geo::cityID();
        }
        $nPerpage = 10;
        $nTotal = $this->db->one_data('SELECT COUNT(I.id) FROM '.TABLE_ITEMS.' I, '.TABLE_ITEMS_ADDR.' IA, '.TABLE_ITEMS_IN_CATEGORIES.' C WHERE '.$sql);
        $aData['pgn'] = $this->generatePagenationDots($nTotal, $nPerpage, 2, static::url('agents.list', 'page={page}'), $sqlLimit);
        $nPage = $this->input->get('page', TYPE_UINT); if (!$nPage) $nPage = 1;

        $aData['items'] = $this->db->select('
                SELECT I.id, I.link, IL.title, IL.descshort as description, IA.address, IA.phonesq as phones, IA.site, I.img_list as img, I.imgcnt
                FROM '.TABLE_ITEMS.' I,
                     '.TABLE_ITEMS_LANG.' IL,
                     '.TABLE_ITEMS_ADDR.' IA,
                     '.TABLE_ITEMS_IN_CATEGORIES.' C
                WHERE '.$sql.$this->db->langAnd(true, 'I', 'IL').'
                GROUP BY I.id 
                ORDER BY I.vip DESC, C.num_order
                '.$sqlLimit);
        
        if ( ! empty($aData['items']))
        {
            foreach($aData['items'] as &$v) {
                if( ! empty($v['phones'])) {
                    $v['phones'] = join('<br/>', explode(', ', $v['phones']));
                }
                $v['link'] = Items::url('view', $v['link']);
            } unset($v);
        }

        # SEO: Агентства - список
        $this->urlCorrection(static::url('agents.list'));
        $this->seo()->canonicalUrl(static::url('agents.list', array(), true), array(
            'page' => $nPage
        ));
        $this->setMeta('agents', array(
            'region' => Geo::filter('title'),
            'page'   => $nPage
        ), $aData);

        bff::setActiveMenu('//job/agents');
        $this->initRightblock(__FUNCTION__);
        return $this->viewPHP($aData, 'agents');
    }

    /**
     * Формируем правый блок
     * @param string $sPage ключ страницы, на которой формируем
     * @param array $aData доп. данные
     */
    protected function initRightblock($sPage, array $aData = array())
    {
        $bAfterBanner = false;

        switch ($sPage)
        {
            case 'index':
            case 'vacancy_index':
            case 'resume_index':
            case 'search':
            {

            } break;
        }
        bff::setRightblock($this->viewPHP($aData, 'common.rightblock'), $bAfterBanner);
    }

    /**
     * Страница успешной активации услуги
     */
    public function promote()
    {
        if( ! bff::servicesEnabled()) return false;

        $nItemID = $this->input->get('id', TYPE_UINT);
        $nSvcID = $this->input->get('svc', TYPE_UINT);

        $aSvc = $this->svc()->model->svcData($nSvcID);
        if( $nItemID && ! empty($aSvc) && $aSvc['module'] == $this->module_name ) {
            bff::setActiveMenu('//job/search', true);
            $aItem = $this->model->itemData($this->typeBySvc($nSvcID), $nItemID, array('id','title','link'));
            if( empty($aItem) ) {
                $this->errors->error404();
            }
            $this->initRightblock(__FUNCTION__);
            bff::setMeta(_t('svc', 'Активация услуги'));
            return $this->showSuccess(
                _t('svc', 'Активация услуги'),
                _t('realty', 'Для вашего объявления "[item_link]"<br />была успешно активирована услуга "[svc_title]"', array(
                    'item_link'=>'<a href="'.Job::url('view', $aItem['link']).'">'.$aItem['title'].'</a>',
                    'svc_title'=>$aSvc['title']))
            );
        } else {
            $this->redirect( static::url('index') );
        }
    }

    /**
     * Крон задачи
     * Рекомендуемый период: раз в час
     */
    public function cron()
    {
        //if( ! bff::cron()) return;
        
        $now = $this->db->now();


        # снимаем с публикации объявления, у которых закончился срок публикации
        $this->db->exec('UPDATE '.TABLE_JOB_VACANCY.' SET
                            status_prev = status, status = '.self::STATUS_PUBLICATED_OUT.'
                       WHERE status = '.self::STATUS_PUBLICATED.' AND publicated_to<=:now', array(':now'=>$now));
        $this->db->exec('UPDATE '.TABLE_JOB_RESUME.' SET
                            status_prev = status, status = '.self::STATUS_PUBLICATED_OUT.'
                       WHERE status = '.self::STATUS_PUBLICATED.' AND publicated_to<=:now', array(':now'=>$now));

        # формируем YandexXML
        $this->yandexXML();

        # выполняем пересчет счетчиков в проф.сферах:
        $this->db->exec('UPDATE '.TABLE_JOB_CATEGORIES.' C
               LEFT JOIN (SELECT I.cat_id as id, COUNT(I.id) as items
                            FROM '.TABLE_JOB_VACANCY.' I, '.TABLE_USERS.' U
                            WHERE I.status = '.self::STATUS_PUBLICATED.' AND I.user_id = U.user_id AND U.blocked = 0
                                  '.(static::premoderation() ? ' AND I.moderated > 0' : '').'
                            GROUP BY I.cat_id) as X ON C.id = X.id
                SET C.vacancy_items = IF(X.id IS NOT NULL, X.items, 0)
        '); 
        $this->db->exec('UPDATE '.TABLE_JOB_CATEGORIES.' C
               LEFT JOIN (SELECT I.cat_id as id, COUNT(I.id) as items
                            FROM '.TABLE_JOB_RESUME.' I, '.TABLE_USERS.' U
                            WHERE I.status = '.self::STATUS_PUBLICATED.' AND I.user_id = U.user_id AND U.blocked = 0
                                  '.(static::premoderation() ? ' AND I.moderated > 0' : '').'
                            GROUP BY I.cat_id) as X ON C.id = X.id
                SET C.resume_items = IF(X.id IS NOT NULL, X.items, 0)
        ');
        
        # выполняем пересчет счетчиков вакансий компаний
        $this->db->exec('UPDATE '.TABLE_ITEMS.' C
               LEFT JOIN (SELECT I.company_id as id, COUNT(I.id) as items
                            FROM '.TABLE_JOB_VACANCY.' I, '.TABLE_USERS.' U
                            WHERE I.company_id !=0 AND I.status = '.self::STATUS_PUBLICATED.' AND I.user_id = U.user_id AND U.blocked = 0
                                '.(static::premoderation() ? ' AND I.moderated > 0' : '').'
                            GROUP BY I.company_id) as X ON C.id = X.id
                SET C.vacancy = IF(X.id IS NOT NULL, X.items, 0)
        ');

        # актуализируем счетчик кол-ва компаний публикующих вакансии в текущий момент
        $nCompany = $this->db->one_data('SELECT I.id
                                            FROM '.TABLE_JOB_VACANCY.' I, '.TABLE_USERS.' U
                                        WHERE I.company_id != 0 AND I.status = '.self::STATUS_PUBLICATED.'
                                          AND I.user_id = U.user_id AND U.blocked = 0
                                          '.(static::premoderation() ? ' AND I.moderated > 0' : '').'
                                        LIMIT 1');
        config::set('job_show_company', !empty($nCompany));
    }
}