<?php

/**
 * Права доступа группы:
 *  - bbs: Доска объявлений
 *      - items-listing: Просмотр списка объявлений
 *      - items-edit: Управление объявлениями (добавление/редактирование/удаление)
 *      - items-moderate: Модерация объявлений (блокирование/одобрение/продление)
 *      - items-comments: Управление комментариями
 *      - claims-listing: Просмотр списка жалоб
 *      - claims-edit: Управление жалобами (модерация/удаление)
 *      - categories: Управление категориями
 *      - types: Управление типами
 *      - svc: Управление услугами
 *
 * Email уведомления:
 *  - bbs_item_blocked - Блокировка объявления
 */

class BBS extends BBSBase
{
    # Объявления
    function items()
    {
        if ( ! $this->haveAccessTo('items-listing'))
            return $this->showAccessDenied();

        $sAct = $this->input->postget('act',TYPE_STR);
        if ( ! empty($sAct) || Request::isPOST())
        {
            $nItemID = $this->input->postget('id', TYPE_UINT);
            if ( ! $this->haveAccessTo('items-edit')) {
                $this->errors->accessDenied();
                $this->ajaxResponseForm();
            }

            $aResponse = array();
            switch ($sAct)
            {
                case 'add':
                {
                    $bSubmit = $this->input->post('save', TYPE_BOOL);
                    $aData = $this->validateItemData(0, $bSubmit);
                    $aResponse['id'] = 0;
                    if ($bSubmit)
                    {

                        if ($this->errors->no())
                        {
                            # публикуем
                            $aData['publicated'] = $this->db->now();
                            $aData['publicated_order'] = $this->db->now();
                            $aData['publicated_to'] = $this->getItemPublicationPeriod();
                            $aData['status'] = self::STATUS_PUBLICATED;
                            $aData['moderated'] = 1;
                            $aData['user_id'] = User::id();

                            $nItemID = $this->model->itemSave(0, $aData, 'd');
                            if ($nItemID > 0) {
                                $aResponse['id'] = $nItemID;
                                $this->itemImages($nItemID)->saveTmp('img');
                                # накручиваем счетчики кол-ва опубликованных объявлений:
                                # в категориях и типах:
                                $aCats = array();
                                for ($i = 1; $i<=3; $i++) { if ( ! empty($aData['cat_id'.$i]) ) $aCats[] = $aData['cat_id'.$i]; }
                                $this->model->itemsCounterUpdate($aCats, array($aData['cat_type']), true, true);
                            }
                            break;
                        }
                    }

                    $aData['id'] = 0;
                    $aData['images'] = array();
                    $aData['imgcnt'] = 0;
                    $aData['img'] = $this->itemImages(0);
                    $aData['user_name'] = $this->security->getUserInfo('name');
                    $aData['user_phone'] = $this->security->getUserInfo('phone');

                    $aData['cats'] = $this->model->categoriesOptionsByLevel( $this->model->categoryParentsID($aData['cat_id']), array('empty'=>_t('','Выбрать')) );
                    $aData['cat'] = $this->model->categoryData($aData['cat_id']);
                    $aData['curr'] = Site::currencyOptions($aData['price_curr']);
                    $aData['cities'] = Geo::cityOptions($aData['city_id'], (!empty($aData['city_id']) ? false : _t('','Выбрать')) );
                    $aResponse['form'] = $this->viewPHP($aData, 'admin.items.form');
                } break;
                case 'edit':
                {
                    $bSubmit = $this->input->post('save', TYPE_BOOL);
                    if ( ! $nItemID) { $this->errors->unknownRecord(); break; }

                    if ($bSubmit)
                    {
                        $aData = $this->validateItemData($nItemID, $bSubmit);

                        if ($this->errors->no()) {
                            $this->model->itemSave($nItemID, $aData, 'd');
                        }
                        break;
                    } else {
                        $aData = $this->model->itemDataEdit($nItemID);
                        if (empty($aData)) { $this->errors->unknownRecord(); break; }
                    }

                    $aData['id'] = $nItemID;

                    # формируем данные о изображениях
                    $aData['img'] = $this->itemImages($nItemID);
                    $aData['images'] = $aData['img']->getData();

                    $this->itemComments()->admListingIncludes(); # js+стили для списка комментариев/жалоб

                    $aData['cats'] = $this->model->categoriesOptionsByLevel( $this->model->categoryParentsID($aData['cat_id']), array('empty'=>'Выбрать'), true );
                    $aData['cat'] = $this->model->categoryData($aData['cat_id']);
                    $aData['dp'] = $this->dpForm($aData['cat_id'], false, $aData);
                    $aData['types'] = $this->model->cattypesByCategory($aData['cat_id'], array('sel'=>$aData['cat_type'],'empty'=>false) );
                    if (empty($aData['types'])) $aData['types'] = false;

                    $aData['curr'] = Site::currencyOptions($aData['price_curr']);
                    $aData['cities'] = Geo::cityOptions($aData['city_id'], (!empty($aData['city_id']) ? false : _t('','Выбрать')) );
                    if ($aData['company_id']) {
                        $aData['aCompanyInfo'] = Users::model()->companyInfo($aData['user_id'], $aData['company_id']);
                    }
                    $aResponse['form'] = $this->viewPHP($aData, 'admin.items.form');
                } break;
                case 'comments-init':
                {
                    $aData['id'] = $nItemID;
                    $aData['edit_allowed'] = $this->haveAccessTo('items-comments');
                    $aData['comments'] = $this->itemComments()->admListing($nItemID);

                    $this->ajaxResponse(array(
                        'html' => $this->viewPHP($aData, 'admin.items.form.comments')
                    ));
                } break;
                case 'claims-init':
                {
                    $aData['id'] = $nItemID;
                    $aData['edit_allowed'] = $this->haveAccessTo('claims-edit');
                    $aData['claims'] = $this->model->claimsListing(array('item_id'=>$nItemID));

                    $this->ajaxResponse(array(
                        'html' => $this->viewPHP($aData, 'admin.items.claims')
                    ));
                } break;
                case 'toggle':
                {

                    $nItemID = $this->input->postget('id', TYPE_UINT);
                    if ( ! $nItemID) { $this->errors->unknownRecord(); break; }

                    $sToggleType = $this->input->get('type', TYPE_STR);

                    $this->model->itemToggle($nItemID, $sToggleType);
                } break;
                case 'category-data':
                {
                    $nCategoryID = $aResponse['id'] = $this->input->post('cat_id', TYPE_UINT);
                    if (empty($nCategoryID)) { $this->errors->unknownRecord(); break; }
                    $bSearch = $this->input->post('search', TYPE_BOOL);

                    $aResponse = $this->model->categoryData($nCategoryID, false);
                    if ($aResponse['subs'] > 0) {
                        $aResponse['cats'] = $this->model->categorySubOptions($nCategoryID, array('sel'=>0, 'empty'=>'Выбрать'));
                    } else {
                        # формируем форму дин. свойств
                        $aResponse['dp'] = $this->dpForm($nCategoryID, $bSearch);
                        # формируем список типов
                        $aResponse['types'] = $this->model->cattypesByCategory($nCategoryID, ( $bSearch ? false : array('sel'=>0,'empty'=>false) ) );
                        if (empty($aResponse['types'])) $aResponse['types'] = false;
                    }

                } break;
                case 'delete':
                {
                    if ( ! $nItemID) { $this->errors->impossible(); break; }

                    $aData = $this->model->itemDataEdit($nItemID);
                    if (empty($aData)) { $this->errors->impossible(); break; }

                    $res = $this->itemDelete($nItemID);
                    if ( ! $res) { $this->errors->impossible(); break; }
                } break;
                case 'items-links-rebuild':
                {
                    if (FORDEV) {
                        $this->model->itemsLinksRebuild();
                    }

                    $this->adminRedirect(Errors::SUCCESS, bff::$event);
                } break;
                case 'items-images-recrop':
                {
                    if (FORDEV) {
                        set_time_limit(0);
                        $this->db->select_rows_chunked(TABLE_BBS_ITEMS, array('id'), array(), 'id', array(), function($items){
                            $images = BBS::i()->itemImages(0);
                            foreach ($items as $v) {
                                $images->setRecordID($v['id']);
                                $images->recropImages(BBSItemImages::szOriginal);
                            }
                        });
                        if ( ! $this->errors->no()) {
                            echo '<pre>', print_r($this->errors->get(1,0), true), '</pre>'; exit;
                        }
                    }

                    $this->adminRedirect(Errors::SUCCESS, bff::$event);
                } break;
                case 'item-user-autocomplete': # autocomplete
                {
                    if ( ! $this->haveAccessTo('items-listing')){
                        $this->ajaxResponse(Errors::ACCESSDENIED);
                    }
                    $sQ = $this->input->post('q', TYPE_NOTAGS);
                    $aData = Users::model()->autocomplete($sQ);
                    $this->autocompleteResponse($aData, 'user_id', 'title');
                }break;
                default: $aResponse = false;
            }

            if ($aResponse!==false && Request::isAJAX()) $this->ajaxResponseForm($aResponse);
        }

        $f = array();
        $this->input->postgetm(array(
            'page'   => TYPE_UINT,
            'cat'    => TYPE_UINT,
            'city'   => TYPE_UINT,
            'tab'    => TYPE_INT,
            'title'  => TYPE_NOTAGS,
            'user'   => TYPE_UINT,
            'company'=> TYPE_UINT,
        ), $f);

        # формируем фильтр списка объявлений
        $sql = array();
        $sqlBind = array();
        $sqlOrder = '';
        $mPerpage = 15;
        $aData['pgn'] = '';

        $aData['orders'] = array('created'=>'desc',);
        $this->prepareOrder($orderBy, $orderDirection, 'created-desc', $aData['orders']);
        $this->tplAssigned(array('order_by', 'order_dir', 'order_dir_needed'), $f);
        $f['order'] = $orderBy.'-'.$orderDirection; $sqlOrder = "$orderBy $orderDirection";

        switch ($f['tab']) {
            case 0: {
                $sql['status'] = self::STATUS_PUBLICATED;
                if(static::premoderation()){
                    $sql[':moderated'] = 'I.moderated > 0';
                }
            } break; # Опубликованные
            case 1: { $sql['status'] = self::STATUS_PUBLICATED_OUT; } break; # Снятые с публикации
            case 2: { $sql[':cond2'] = '(I.moderated != 1 AND I.status NOT IN('.self::STATUS_PUBLICATED_OUT.','.self::STATUS_NOTACTIVATED.'))'; } break; # На модерации
            case 3: { $sql['status'] = self::STATUS_BLOCKED; } break; # Заблокированные
            case 6: { $sql['deleted'] = 1; } break; # Помеченные на удаление пользователем
            case 4: break; # Все
        }

        if ($f['cat']>0) {
            $sql[':cat_id'] = array('(I.cat_id3 = :cat OR I.cat_id2 = :cat OR I.cat_id1 = :cat)',
                                ':cat'=>$f['cat']);
        }
        if ($f['city'] > 0) {
            $sql[':city_id'] = 'I.city_id = '.$f['city'];
        }
        if (!empty($f['title'])) {
            $sql[':title'] = array('(I.id = '.intval($f['title']).' OR I.title LIKE :title)',
                                   ':title'=>'%'.$f['title'].'%');
        }
        if ($f['user'] > 0) {
            $sql['user_id'] = $f['user'];
            $aUser = Users::model()->userData($f['user'], array('login', 'email'));
            $aData['user_title'] = '#'.$f['user'].' '.$aUser['login'].' ('.$aUser['email'].')';
        }
        if ($f['company'] > 0) {
            $sql['company_id'] = $f['company'];
        }

        if ($mPerpage!==false) {
            $nCount = $this->model->itemsListing($sql, true, $sqlBind);
            $aData['pgn'] = $this->generatePagenation($nCount, $mPerpage, 'jBbsItemsList.page({pageId})', $sqlLimit, 'pagenation.ajax.tpl', 'page', true);
            $aData['list'] = $this->model->itemsListing($sql, false, $sqlBind, $sqlLimit, $sqlOrder);
        } else {
            $aData['list'] = $this->model->itemsListing($sql, false, $sqlBind, '', $sqlOrder);
        }

        $aData['list'] = $this->viewPHP($aData, 'admin.items.listing.ajax');

        if (Request::isAJAX()) {
            $this->ajaxResponseForm(array(
                'list' => $aData['list'],
                'pgn'  => $aData['pgn'],
            ));
        }

        $aData['f'] = $f;
        $aData['id'] = $this->input->get('id', TYPE_UINT);
        $aData['act'] = $sAct;

        $aData['cats'] = $this->model->categoriesOptions($f['cat'], array('Все категории'));
        $this->itemComments()->admListingIncludes();
        tpl::includeJS(array('ui.sortable','qquploader', 'autocomplete'), true);
        Geo::mapsAPI(true);

        return $this->viewPHP($aData, 'admin.items.listing');
    }

    function ajax()
    {
        if ( ! $this->haveAccessTo('items-moderate')) {
            $this->errors->accessDenied();
            $this->ajaxResponseForm();
        }

        $nItemID = $this->input->post('id', TYPE_UINT);
        if ( ! $nItemID) {
            $this->errors->unknownRecord();
            $this->ajaxResponseForm();
        }

        $aResponse = array();
        switch ($this->input->get('act', TYPE_STR))
        {
            case 'item-block':
            {

                /**
                 * Блокировка объявления (если уже заблокировано => изменение причины блокировки)
                 * @param string 'blocked_reason' причина блокировки
                 * @param integer 'id' ID объявления
                 */

                $sBlockedReason  = $this->input->postget('blocked_reason', TYPE_NOTAGS, array('len'=>1000));

                $aData = $this->model->itemData($nItemID, array('status','deleted','user_id'));
                if (empty($aData)) { $this->errors->unknownRecord(); break; }

                $bBlocked = ($aData['status'] == self::STATUS_BLOCKED);

                $aUpdate = array (
                    'moderated' => 1,
                    'blocked_reason' => $sBlockedReason,
                );

                if ( ! $bBlocked) {
                    $aUpdate[] = 'blocked_num = blocked_num + 1';
                    $aUpdate[] = 'status_prev = status';
                    $aUpdate['status'] = self::STATUS_BLOCKED;
                }

                $res = $this->model->itemSave($nItemID, $aUpdate);
                if ($res && !$bBlocked)
                {
                    $bBlocked = true;

                    # отправляем email-уведомление о блокировке объявления
                    do
                    {
                        if ($aData['status'] == self::STATUS_NOTACTIVATED) break;
                        if ( !empty($aData['deleted'])) break;
                        if (empty($aData['user_id'])) break;

                        $aDataEmail = $this->model->itemData2Email($nItemID);
                        if ($aDataEmail === false) break;

                        bff::sendMailTemplate(
                            array('name'=>$aDataEmail['name'],
                                  'email'=>$aDataEmail['email'],
                                  'item_id'=>$aDataEmail['item_id'],
                                  'item_link'=>$aDataEmail['item_link'],
                                  'item_title'=>$aDataEmail['item_title'],
                                  'blocked_reason'=>$sBlockedReason
                                  ),
                            'bbs_item_blocked', $aDataEmail['email']);

                    } while(false);
                }

                $aResponse['blocked'] = $bBlocked;
                $aResponse['reason']  = $sBlockedReason;
            } break;
            case 'item-approve':
            {

                $aItemData = $this->model->itemData($nItemID, array('status', 'publicated', 'publicated_to', 'cat_id', 'cat_id1', 'cat_id2', 'cat_id3', 'cat_type'));
                if (empty($aItemData) || $aItemData['status'] == self::STATUS_NOTACTIVATED) {
                    $this->errors->unknownRecord(); break;
                }

                $aUpdate = array(
                    'moderated'=>1
                );

                if ($aItemData['status'] == self::STATUS_BLOCKED) {
                    /**
                     * В случае если "Одобряем" заблокированное объявление
                     * => значит оно после блокировки было отредактировано пользователем
                     * => следовательно если его период публикации еще не истек => Публикуем,
                     *    в противном случае переводим в статус "Период публикации завершился"
                     */
                    $newStatus = self::STATUS_PUBLICATED_OUT;
                    $now = time();
                    $from = strtotime($aItemData['publicated']);
                    $to = strtotime($aItemData['publicated_to']);
                    if ( ! empty($from) && !empty($to) && $now >= $from && $now < $to) {
                        $newStatus = self::STATUS_PUBLICATED;
                    }
                    $aUpdate[] = 'status_prev = status';
                    $aUpdate['status'] = $newStatus;
                }

                $this->model->itemSave($nItemID, $aUpdate);

                if(static::premoderation()){
                    $aCats = array();
                    for($i = 1; $i<=3; $i++) { if ( ! empty($aItemData['cat_id'.$i]) ) $aCats[] = $aItemData['cat_id'.$i]; }
                    $this->model->itemsCounterUpdate($aCats, array($aItemData['cat_type']), true, true);
                }

            } break;
            case 'item-publicate2':
            {

                /**
                 * Продление публикации объявления
                 */
                $aItem = $this->model->itemData($nItemID, array('id', 'status', 'moderated',
                                'publicated', 'publicated_to',
                                'cat_id1', 'cat_id2', 'cat_id3', 'cat_type'));
                if (empty($aItem)) {
                    $this->errors->unknownRecord(); break;
                }

                $nPeriod = $this->input->post('period', TYPE_UINT);

                switch ($aItem['status'])
                {
                    case self::STATUS_NOTACTIVATED: {
                        $this->errors->set('Невозможно продлить публикацию неактивированного объявления');
                    } break;
                    case self::STATUS_BLOCKED: {
                        $this->errors->set('Невозможно продлить публикацию, поскольку объявление '.
                            ($aItem['moderated'] == 0 ? 'ожидает проверки' : 'отклонено') );
                    } break;
                    case self::STATUS_PUBLICATED: {
                        $this->errors->set('Невозможно продлить публикацию, поскольку объявление опубликовано');
                        // продление опубликованных пока НЕ делаем
                        // $publicateTo = $this->getPublicatePeriodDate( false, strtotime(BFF_NOW) ); // от текущего момента
                        // $this->model->itemSave( $nItemID, array('publicated_to' => $publicateTo) );
                    } break;
                    case self::STATUS_PUBLICATED_OUT:
                    {
                        $aUpdate = array(
                            'publicated_to' => $this->getItemRefreshPeriod(), # помечаем дату окончания публикации
                            'status_prev = status',
                            'status' => self::STATUS_PUBLICATED,
                            'moderated' => 1,
                        );

                        /**
                         * Если разница между датой снятия с публикации и текущей датой
                         * более 3 дней, тогда поднимаем объявление вверх.
                         * в противном случае: оставлем дату старта публикации(pulicated)
                         *   и дату порядка публикации(publicated_order) прежними
                         */
                        if ( ((BFF_NOW - strtotime($aItem['publicated_to'])) > 259200) ) { # 3 суток
                            $aUpdate['publicated'] = $this->db->now();
                            $aUpdate['publicated_order'] = $this->db->now();
                        }

                        $res = $this->model->itemSave($nItemID, $aUpdate);
                        if ( ! empty($res)) {
                            # накручиваем счетчики кол-ва опубликованных объявлений:
                            # в категориях и типах:
                            $this->model->itemsCounterUpdate(array($aItem['cat_id1'], $aItem['cat_id2'], $aItem['cat_id3']),
                                                      (!empty($aItem['cat_type']) ? array($aItem['cat_type']) : array()),
                                                      true, true);
                        }
                    } break;
                }
            } break;
            case 'item-unpublicate':
            {

                $aItem = $this->model->itemData($nItemID, array('id', 'status', 'moderated',
                                'publicated', 'publicated_to', 'cat_id1', 'cat_id2', 'cat_id3', 'cat_type'));
                if (empty($aItem) || $aItem['status'] != self::STATUS_PUBLICATED) {
                    $this->errors->impossible(); break;
                }

                $sqlDateEmpty = '0000-00-00 00:00:00';
                $aUpdate = array(
                    'status_prev = status',
                    'status' => self::STATUS_PUBLICATED_OUT,
                    'moderated' => 1,
                    'publicated_to' => $this->db->now(),
                    # оставляем все текущие услуги активированными
                    //'svc' => 0,
                    //'premium_to' => $sqlDateEmpty,
                    //'marked_to' => $sqlDateEmpty,
                );

                $res = $this->model->itemSave($nItemID, $aUpdate);
                if (empty($res)) {
                    # ПЕРЕСЧИТЫВАЕМ счетчики в категориях/типах ПО КРОНУ
                    $this->errors->impossible();
                }
            } break;

        }

        $this->ajaxResponseForm($aResponse);
    }

    # Фотографии объявления
    function img()
    {
        if ( ! $this->haveAccessTo('items-edit') )
            return $this->showAccessDenied();

        $nItemID = $this->input->get('item_id', TYPE_UINT);

        $oImages = $this->itemImages($nItemID);
        $sAction = $this->input->getpost('act');

        switch ($sAction)
        {
            case 'upload':
            {
                $result = $oImages->uploadQQ();

                $aResponse = array( 'success'=> ($result !== false && $this->errors->no()) );
                if ($result !== false) {
                    $aResponse = array_merge($aResponse, $result);
                    $aURL = $oImages->getURL($result, array(BBSItemImages::szMini, BBSItemImages::szView), empty($nItemID));
                    $aResponse = array_merge($aResponse, $aURL);
                }
                $aResponse['errors'] = $this->errors->get(true);
                $this->ajaxResponse($aResponse, true, false, true);
            } break;
            case 'saveorder':
            {
                $img = $this->input->post('img', TYPE_ARRAY);
                if ( ! $oImages->saveOrder($img, false, true)) {
                    $this->errors->impossible();
                }
            } break;
            case 'delete':
            {
                $nImageID = $this->input->post('image_id', TYPE_UINT);
                $sFilename = $this->input->post('filename', TYPE_STR);
                if ( ! $nImageID && empty($sFilename)) {
                    $this->errors->impossible(); break;
                }
                if ($nImageID) {
                    $oImages->deleteImage($nImageID);
                } else {
                    $oImages->deleteTmpFile($sFilename);
                }
            } break;
            case 'delete-all':
            {
                if ($nItemID) {
                    $oImages->deleteAllImages(true);
                } else {
                    $aFilenames = $this->input->post('filenames', TYPE_ARRAY_STR);
                    $oImages->deleteTmpFile($aFilenames);
                }
            } break;
            default:
            {
                $this->errors->impossible();
            } break;
        }

        $aResponse['success'] = $this->errors->no();
        $this->ajaxResponse($aResponse);
    }

    //-------------------------------------------------------------------------------------------------------------------------------
    // Жалобы

    function claims()
    {
        if ( ! $this->haveAccessTo('claims-listing') )
            return $this->showAccessDenied();

        $sNewCounterKey = 'bbs_claims';

        if (Request::isAJAX())
        {
            switch ($this->input->get('act', TYPE_STR))
            {
                case 'delete': # удаляем жалобу
                {
                    if ( ! $this->haveAccessTo('claims-edit'))
                        $this->ajaxResponse(Errors::ACCESSDENIED);

                    $nClaimID = $this->input->post('claim_id', TYPE_UINT);
                    if ($nClaimID)
                    {
                        $aData = $this->model->claimData($nClaimID, array('id', 'item_id', 'viewed'));
                        if (empty($aData)) $this->ajaxResponse(Errors::IMPOSSIBLE);

                        $aResponse = array('counter_update'=>false);
                        $res = $this->model->claimDelete($nClaimID);
                        if ($res && !$aData['viewed']) {
                            # общий счетчик
                            config::saveCount($sNewCounterKey, -1, true);
                            $aResponse['counter_update'] = true;
                        }
                        # счетчик жалоб у ОБ
                        $this->model->itemSave($aData['item_id'], array(
                            'claims_cnt = claims_cnt - 1'
                        ));

                        $aResponse['res'] = $res;
                        $this->ajaxResponse($aResponse);
                    }
                } break;
                case 'viewed': # отмечаем жалобу как прочитанную
                {
                    if ( ! $this->haveAccessTo('claims-edit'))
                        $this->ajaxResponse(Errors::ACCESSDENIED);

                    $nClaimID = $this->input->post('claim_id', TYPE_UINT);
                    if ($nClaimID) {
                        $res = $this->model->claimSave($nClaimID, array('viewed'=>1));
                        if ($res) config::saveCount($sNewCounterKey, -1, true);
                        $this->ajaxResponse(Errors::SUCCESS);
                    }
                } break;
                case 'item-users-autocomplete': # autocomplete
                {
                    if (!$this->haveAccessTo('items-listing')) {
                        $this->ajaxResponse(Errors::ACCESSDENIED);
                    }
                    $sQ = $this->input->post('q', TYPE_NOTAGS);
                    $aData = Users::model()->autocomplete($sQ);
                    $this->autocompleteResponse($aData, 'user_id', 'title');
                }
                    break;

            }
            $this->ajaxResponse(Errors::IMPOSSIBLE);
        }

        $aData = $this->input->getm(array(
            'item'    => TYPE_UINT,
            'page'    => TYPE_UINT,
            'perpage' => TYPE_UINT,
            'user'    => TYPE_UINT,
        ));

        $aFilter = array();
        if ($aData['item']) {
            $aFilter['item_id'] = $aData['item'];
        }
        if ($aData['user']) {
            $aFilter['user_id'] = $aData['user'];
        }

        $nCount = $this->model->claimsListing($aFilter, true);

        $aPerpage = $this->preparePerpage($aData['perpage'], array(20,40,60));

        $sFilter = http_build_query($aData); unset($aData['page']);
        $aData['pgn'] = $this->generatePagenation($nCount, $aData['perpage'], $this->adminLink("claims&$sFilter&{pageId}"), $sqlLimit);

        $aData['list'] = ( $nCount > 0 ?
                $this->model->claimsListing($aFilter, false, $sqlLimit) :
                array() );
        $aData['list'] = $this->viewPHP($aData, 'admin.claims.list');

        if($aData['user']){
            $aUser = Users::model()->userData($aData['user'], array('login', 'email'));
            $aData['user_title'] = '#'.$aData['user'].' '.$aUser['login'].' ('.$aUser['email'].')';
        }

        $aData['perpage'] = $aPerpage;
        return $this->viewPHP($aData, 'admin.claims');
    }

    //-------------------------------------------------------------------------------------------------------------------------------
    // Комментарии

    function comments_ajax()
    {
        if ( ! $this->haveAccessTo('items-comments'))
            return $this->showAccessDenied();

        $this->itemComments()->admAjax();
    }

    function comments_mod()
    {
        if ( ! $this->haveAccessTo('items-comments'))
            return $this->showAccessDenied();

        $aData['comments'] = $this->itemComments()->admListingModerate(15);

        return  $this->viewPHP($aData, 'admin.items.comments.mod');
    }

    //-------------------------------------------------------------------------------------------------------------------------------
    // Категории

    function categories()
    {
        if ( ! $this->haveAccessTo('categories'))
            return $this->showAccessDenied();

        $sAct = $this->input->postget('act',TYPE_STR);
        if ( ! empty($sAct) || Request::isPOST() )
        {

            $aResponse = array();
            switch ($sAct)
            {
                case 'add':
                {
                    $bSubmit = $this->input->post('save', TYPE_BOOL);
                    $aData = $this->validateCategoryData(0, $bSubmit);
                    if ($bSubmit)
                    {
                        if ($this->errors->no()) {
                            $nCategoryID = $this->model->categorySave(0, $aData);
                            if ($nCategoryID > 0) {
                                $this->adminRedirect(Errors::SUCCESS, 'categories');
                            }
                        }
                    }

                    $aData['id'] = 0;

                    $aResponse['form'] = $this->viewPHP($aData, 'admin.categories.form');
                } break;
                case 'edit':
                {
                    $bSubmit = $this->input->post('save', TYPE_BOOL);
                    $nCategoryID = $this->input->postget('id', TYPE_UINT);
                    if ( ! $nCategoryID) { $this->errors->unknownRecord(); break; }

                    if ($bSubmit)
                    {
                        $aData = $this->validateCategoryData($nCategoryID, $bSubmit);
                        if ($this->errors->no()) {
                            $this->model->categorySave($nCategoryID, $aData);
                            $this->adminRedirect(Errors::SUCCESS, ($this->input->post('return', TYPE_BOOL) ? 'categories' : 'categories&act=edit&id='.$nCategoryID) );
                        }
                        $aData['id'] = $nCategoryID;
                    } else {
                        $aData = $this->model->categoryData($nCategoryID, true);
                        if (empty($aData)) { $this->errors->unknownRecord(); break; }
                    }

                    $aData['pid_path'] = $this->model->categoryParentsTitle($nCategoryID);
                    $aResponse['form'] = $this->viewPHP($aData, 'admin.categories.form');
                } break;
                case 'expand':
                {
                    $nCategoryID = $this->input->postget('id', TYPE_UINT);
                    if ( ! $nCategoryID) { $this->errors->unknownRecord(); break; }

                    $aData['list'] = $this->model->categoriesListing(array('pid'=>$nCategoryID));
                    $aData['skip_norecords'] = false;
                    $aResponse['list'] = $this->viewPHP($aData, 'admin.categories.listing.ajax');
                    $aResponse['cnt']  = sizeof($aData['list']);
                } break;
                case 'toggle':
                {
                    $nCategoryID = $this->input->postget('id', TYPE_UINT);
                    if ( ! $nCategoryID) { $this->errors->unknownRecord(); break; }

                    $sToggleType = $this->input->get('type', TYPE_STR);

                    $this->model->categoryToggle($nCategoryID, $sToggleType);
                } break;
                case 'rotate':
                {
                    $this->model->categoriesRotate();
                } break;
                case 'delete':
                {
                    $nCategoryID = $this->input->postget('id', TYPE_UINT);
                    if ( ! $nCategoryID) { $this->errors->impossible(); break; }

                    $aData = $this->model->categoryData($nCategoryID, true);
                    if (empty($aData)) { $this->errors->impossible(); break; }

                    $this->model->categoryDelete($nCategoryID);

                } break;
                case 'dev-ns-valid':
                {
                    # Валидация целостности NestedSets категорий
                    if ( ! FORDEV) return $this->showAccessDenied();
                    return $this->model->treeCategories->validate(true);
                } break;
                default: $aResponse = false;
            }

            if ($aResponse!==false && Request::isAJAX()) $this->ajaxResponseForm($aResponse);
        }

        $f = array();
        $this->input->postgetm(array(
            'page'   => TYPE_UINT,
        ), $f);

        # формируем фильтр списка категорий
        $sql = array();
        $aData['pgn'] = '';

        $sExpandState = $this->input->cookie(config::sys('cookie.prefix').'bbs_categories_expand', TYPE_STR);
        $aExpandID = ( ! empty($sExpandState) ? explode('.', $sExpandState) : array());
        $aExpandID = array_map('intval', $aExpandID); $aExpandID[] = 1;
        $sql[] = 'pid IN ('.join(',', $aExpandID).')';

        $aData['list'] = $this->model->categoriesListing($sql);

        $aData['list'] = $this->viewPHP($aData, 'admin.categories.listing.ajax');

        if (Request::isAJAX()) {
            $this->ajaxResponseForm(array(
                'list' => $aData['list'],
                'pgn'  => $aData['pgn'],
            ));
        }

        $aData['f'] = $f;
        $aData['id'] = $this->input->get('id', TYPE_UINT);
        $aData['act'] = $sAct;

        return $this->viewPHP($aData, 'admin.categories.listing');
    }

    /**
     * Обрабатываем параметры запроса
     * @param integer $nCategoryID ID категории или 0
     * @param boolean $bSubmit выполняем сохранение/редактирование
     * @return array параметры
     */
    protected function validateCategoryData($nCategoryID, $bSubmit)
    {
        $aData = $this->input->postm(array(
            'pid'       => TYPE_UINT, # Основной раздел
            'addr'      => TYPE_BOOL, # Точка на карте
            'prices'    => TYPE_BOOL, # Цена
            'mtemplate' => TYPE_BOOL, # SEO
        ));
        if ( ! $nCategoryID) { # URL-Keyword
            $aData['keyword'] = $this->input->post('keyword', TYPE_NOTAGS);
        }

        $this->input->postm_lang($this->model->langCategories, $aData);

        if ($bSubmit)
        {
            if ( ! $aData['pid']) {
                $this->errors->set(_t('bbs','Укажите основную категорию'));
            }

            if (empty($aData['title'][LNG])) {
                $this->errors->set(_t('bbs','Укажите название категории'));
            }

            if ( ! $nCategoryID)
            {
                # формируем keyword
                $sKeyword = trim($aData['keyword'], '/ ');
                $sTitle   = $aData['title'][LNG];

                if (empty($sKeyword) && ! empty($sTitle)) {
                    $sKeyword = mb_strtolower(func::translit($sTitle));
                }
                $sKeyword = preg_replace('/[^a-z0-9_\-]/','',$sKeyword);

                do{

                    if (empty($sKeyword)) {
                        $this->errors->set(_t('bbs','Keyword указан некорректно'));
                        break;
                    }

                    # parent-keyword / ... / keyword
                    $sKeywordFull = array();
                    if ($aData['pid'] > 1) {
                        $aCats = $this->model->categoryParentsData($aData['pid'], true, true);
                        if (empty($aCats)) {
                            $this->errors->set(_t('bbs','Основная категория указана некорректно'));
                            break;
                        }
                        foreach ($aCats as $v) {
                            $sKeywordFull[] = $v['keyword'];
                        }
                    }
                    $sKeywordFull[] = $sKeyword;
                    $aData['keyword'] = join('/', $sKeywordFull);

                } while(false);
            }
        }
        return $aData;
    }

    //-------------------------------------------------------------------------------------------------------------------------------
    // Типы категорий

    public function types()
    {
        if ( ! $this->haveAccessTo('types'))
            return $this->showAccessDenied();

        $nCategoryID = $this->input->getpost('cat_id', TYPE_UINT);
        if ( ! $nCategoryID) $this->adminRedirect(Errors::IMPOSSIBLE, 'categories_listing');

        if (Request::isAJAX())
        {
            switch ($this->input->get('act', TYPE_STR))
            {
                case 'toggle':
                {

                    $nTypeID = $this->input->get('type_id', TYPE_UINT);
                    if ($this->model->cattypeToggle($nTypeID, 'enabled')) {
                        $this->ajaxResponse(Errors::SUCCESS);
                    }
                } break;
                case 'rotate':
                {

                    if ($this->model->cattypesRotate($nCategoryID))
                        $this->ajaxResponse(Errors::SUCCESS);
                } break;
                case 'form':
                {
                    $nTypeID = $this->input->get('type_id', TYPE_UINT);
                    if ($nTypeID) {
                        $aData = $this->model->cattypeData($nTypeID, '*', true);
                    } else {
                        $this->validateCategoryTypeData($aData, $nCategoryID, 0);
                        $aData['id'] = 0;
                    }

                    $aData['form'] = $this->viewPHP($aData, 'admin.types.form');
                    $this->ajaxResponse($aData);
                } break;
                case 'delete':
                {

                    $nTypeID = $this->input->get('type_id', TYPE_UINT);
                    if ($this->model->cattypeDelete($nTypeID)) {
                        $this->ajaxResponse(Errors::SUCCESS);
                    }
                } break;
            }
            $this->ajaxResponse(Errors::IMPOSSIBLE);
        }

        if (Request::isPOST())
        {

            switch ($this->input->postget('action', TYPE_STR))
            {
                case 'add':
                {
                    $this->validateCategoryTypeData($aData, $nCategoryID, 0);

                    if ($this->errors->no())
                    {
                        $res = $this->model->cattypeSave(0, $nCategoryID, $aData);
                        $this->adminRedirect(($res ? Errors::SUCCESS : Errors::IMPOSSIBLE), 'types&cat_id='.$nCategoryID);
                    }
                } break;
                case 'edit':
                {
                    $nTypeID = $this->input->post('type_id', TYPE_UINT);
                    if ( ! $nTypeID) {
                        $this->adminRedirect(Errors::IMPOSSIBLE, 'types&cat_id='.$nCategoryID);
                    }

                    $this->validateCategoryTypeData($aData, $nCategoryID, $nTypeID );

                    if ($this->errors->no())
                    {
                        $res = $this->model->cattypeSave($nTypeID, $nCategoryID, $aData);

                        $this->adminRedirect(($res ? Errors::SUCCESS : Errors::IMPOSSIBLE), 'types&cat_id='.$nCategoryID);
                    }
                } break;
            }
        }

        $aData['cats'] = $this->model->categoryParentsData($nCategoryID, true, true);

        $aData['types'] = $this->model->cattypesListing( array($this->db->prepareIN('T.cat_id', array_keys($aData['cats']))) );

        $aData['url_listing'] = $this->adminLink('types');
        $aData['url_action'] = $this->adminLink('types&cat_id='.$nCategoryID.'&act=');
        $aData['cat_id'] = $nCategoryID;

        tpl::includeJS(array('tablednd'), true);
        return $this->viewPHP($aData, 'admin.types.listing');
    }

    /**
     * Обработка данных типа категории
     * @param array $aData @ref данные
     * @param integer $nCategoryID ID категории
     * @param integer $nTypeID ID типа
     */
    protected function validateCategoryTypeData(&$aData, $nCategoryID, $nTypeID)
    {
        $this->input->postm_lang($this->model->langCategoriesTypes, $aData );

        if (Request::isPOST())
        {
            if ($this->errors->no())
            {
                //
            }
        }
    }

    public function settings()
    {
        if (!$this->haveAccessTo('settings')) {
            return $this->showAccessDenied();
        }

        $sCurrentTab = $this->input->postget('tab');
        if (empty($sCurrentTab)) {
            $sCurrentTab = 'general';
        }

        $aLang = array(
            'form_add'  => TYPE_STR,
            'form_edit' => TYPE_STR,
        );

        if (Request::isPOST() && $this->input->post('save', TYPE_BOOL)) {

            $aData = $this->input->postm(array(
                    'item_publication_period'         => TYPE_UINT,
                    'item_refresh_period'             => TYPE_UINT,
                    'item_delete_unactivate_period'   => TYPE_UINT,
                )
            );

            # срок публикации
            if (!$aData['item_publication_period']) {
                $aData['item_publication_period'] = 30;
            } else {
                if ($aData['item_publication_period'] > 100) {
                    $aData['item_publication_period'] = 100;
                }
            }

            # срок продления
            if (!$aData['item_refresh_period']) {
                $aData['item_refresh_period'] = 30;
            } else {
                if ($aData['item_refresh_period'] > 100) {
                    $aData['item_refresh_period'] = 100;
                }
            }

            # срок удаления не активированных, часов
            if (!$aData['item_delete_unactivate_period']) {
                $aData['item_delete_unactivate_period'] = 24;
            } else {
                if ($aData['item_delete_unactivate_period'] > 100) {
                    $aData['item_delete_unactivate_period'] = 100;
                }
            }

            $this->input->postm_lang($aLang, $aData);
            $this->db->langFieldsModify($aData, $aLang, $aData);
            $this->configSave($aData);

            $this->adminRedirect(Errors::SUCCESS, 'settings&tab=' . $sCurrentTab);
        }

        $aData = $this->configLoad();
        foreach ($this->locale->getLanguages() as $lng) {
            foreach ($aLang as $k => $v) {
                if (!isset($aData[$k . '_' . $lng])) {
                    $aData[$k . '_' . $lng] = '';
                }
            }
        }

        $aData['tab'] = $sCurrentTab;
        $aData['tabs'] = array(
            'general' => array('t' => 'Общие настройки'),
        );

        return $this->viewPHP($aData, 'admin.settings');
    }


    //-------------------------------------------------------------------------------------------------------------------------------
    // Услуги

    public function svc_services()
    {
        if ( ! $this->haveAccessTo('svc'))
            return $this->showAccessDenied();

        $svc = $this->svc()->model;

        if (Request::isAJAX())
        {
            $aResponse = array();

            switch ($this->input->getpost('act'))
            {
                case 'update':
                {
                    $nSvcID = $this->input->post('id', TYPE_UINT);
                    if ( ! $nSvcID) {
                        $this->errors->unknownRecord();
                        break;
                    }

                    $aData = $svc->svcData($nSvcID, array('id','type','keyword'));
                    if (empty($aData) || $aData['type'] != Svc::TYPE_SERVICE) {
                        $this->errors->unknownRecord();
                        break;
                    }

                    $this->svcValidateData($nSvcID, Svc::TYPE_SERVICE, $aDataSave);

                    if ($this->errors->no()) {
                        $svc->svcSave($nSvcID, $aDataSave);
                        $aResponse['modified'] = tpl::date_format2(BFF_NOW, true).', <a class="bold desc" href="#" onclick="return bff.userinfo('.User::id().');">'.User::data('login').'</a>';
                    }

                } break;
                default: {
                    $this->errors->impossible();
                } break;
            }

            $this->ajaxResponseForm($aResponse);
        }

        $aData = array(
            'svc' => $svc->svcListing(Svc::TYPE_SERVICE, $this->module_name, array('premium','press')),
        );

        tpl::includeJS(array('wysiwyg'), true);
        return $this->viewPHP($aData, 'admin.services');
    }

    /**
     * Проверка данных услуги / пакета услуг
     * @param integer $nSvcID ID услуги / пакета услуг
     * @param integer $nType тип Svc::TYPE_
     * @param array $aData @ref проверенные данные
     */
    protected function svcValidateData($nSvcID, $nType, &$aData)
    {
        $aParams = array(
            'price' => TYPE_PRICE
        );

        if ($nType == Svc::TYPE_SERVICE)
        {
            $aSettings = array(
                'on' => TYPE_BOOL, # включена
            );

            $aSettingsLang = array(
                'description' => TYPE_STR, # описание
            );

            switch ($nSvcID)
            {
                case self::SERVICE_UP:
                {

                } break;
                case self::SERVICE_MARK:
                {

                } break;
                case self::SERVICE_FIX:
                {

                } break;
                case self::SERVICE_PREMIUM:
                {

                } break;
                case self::SERVICE_PRESS:
                {

                } break;
                case self::SERVICE_VIP:
                {
                    $aSettings['cnt'] = TYPE_UINT;
                } break;
            }

            $aData = $this->input->postm($aParams);
            $aData['settings'] = $this->input->postm($aSettings);
            $this->input->postm_lang($aSettingsLang, $aData['settings']);

            if (Request::isPOST())
            {
                //
            }
        }
    }

}